from PyQt5.QtCore import Qt, QRegExp
from PyQt5.QtGui import QRegExpValidator
from PyQt5.QtWidgets import (QMainWindow, QVBoxLayout, QPushButton, QSizePolicy, QHBoxLayout, QLabel, QCheckBox,
                             QStackedWidget, QProgressBar, QLineEdit, QWidget, QGroupBox)

from doocspie.pyqt import messages
from ..plotter.viewer_plotter import ViewerPlotter


class ViewerToolUi(QMainWindow):
    START_LABEL = "Start"

    def __init__(self, roi_data):
        super().__init__()
        self._name = "Viewer"
        self._roi_data = roi_data
        self._main_layout = QVBoxLayout()
        self._viewer_plotter = ViewerPlotter()
        self._start_button = self._get_labeled_push_button(self.START_LABEL)
        self._stop_button = self._get_labeled_push_button("Stop")
        self._lock_aspect_ratio_checkbox = QCheckBox("Lock Aspect Ratio")
        self._take_button_and_progress_bar_stack = QStackedWidget()
        self._take_button = self._get_labeled_push_button("Take")
        self._progress_bar = QProgressBar()
        self._samples_line_edit = self._get_samples_line_edit()
        self._apply_background_correction_check_box = QCheckBox("Apply", enabled=False)
        self._roi_widgets_layout = QHBoxLayout()
        self._enable_roi_checkbox = QCheckBox("Enable")
        self._reset_roi_button = self._get_labeled_push_button("Reset")
        self._save_roi_button = self._get_labeled_push_button("Save")
        self._clear_roi_button = self._get_labeled_push_button("Clear")
        self._roi_x_position_label = None
        self._roi_x_position_line_edit = None
        self._roi_y_position_label = None
        self._roi_y_position_line_edit = None
        self._roi_width_label = None
        self._roi_width_line_edit = None
        self._roi_height_label = None
        self._roi_height_line_edit = None
        self._create_layout()
        self._create_connections()
        self._image_read_once = False

    @staticmethod
    def _get_labeled_push_button(label):
        push_button = QPushButton(label)
        push_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        return push_button

    @staticmethod
    def _get_samples_line_edit():
        line_edit = QLineEdit("10")
        line_edit.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setMaxLength(3)
        line_edit.setMaximumWidth(33)
        line_edit.setValidator(QRegExpValidator(QRegExp("[1-9][0-9]*")))
        return line_edit

    @property
    def roi(self):
        return self._viewer_plotter.roi

    def _create_layout(self):
        self._create_main_layout()
        self._create_plot_layout()
        self._create_controls_layout()

    def _create_main_layout(self):
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_plot_layout(self):
        plot_layout = QHBoxLayout()
        plot_layout.addWidget(self._viewer_plotter.widget)
        self._main_layout.addLayout(plot_layout)

    def _create_controls_layout(self):
        controls_layout = QHBoxLayout()
        controls_layout.addWidget(self._create_control_group_box())
        controls_layout.addWidget(self._create_background_correction_group_box())
        controls_layout.addWidget(self._create_region_of_interest_group_box())
        self._main_layout.addLayout(controls_layout)

    def _create_control_group_box(self):
        group_box = QGroupBox("Control")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._start_button)
        widgets_layout.addWidget(self._stop_button)
        widgets_layout.addWidget(self._lock_aspect_ratio_checkbox)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_background_correction_group_box(self):
        group_box = QGroupBox("Background Correction")
        group_box.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        widgets_layout = QHBoxLayout()
        self._take_button_and_progress_bar_stack.addWidget(self._take_button)
        self._take_button_and_progress_bar_stack.addWidget(self._progress_bar)
        widgets_layout.addWidget(self._take_button_and_progress_bar_stack)
        widgets_layout.addWidget(QLabel("Samples:"))
        widgets_layout.addWidget(self._samples_line_edit)
        widgets_layout.addWidget(self._apply_background_correction_check_box)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_region_of_interest_group_box(self):
        group_box = QGroupBox("Region of Interest")
        self._roi_widgets_layout.addWidget(self._enable_roi_checkbox)
        self._enable_roi_checkbox.setEnabled(False)
        self._roi_widgets_layout.addWidget(self._reset_roi_button)
        self._reset_roi_button.setEnabled(False)
        self._roi_widgets_layout.addWidget(self._save_roi_button)
        self._save_roi_button.setEnabled(False)
        self._roi_widgets_layout.addWidget(self._clear_roi_button)
        self._clear_roi_button.setEnabled(False)
        group_box.setLayout(self._roi_widgets_layout)
        return group_box

    def _create_connections(self):
        self._lock_aspect_ratio_checkbox.stateChanged.connect(self._viewer_plotter.lock_aspect_ratio)
        self._lock_aspect_ratio_checkbox.stateChanged.connect(self._viewer_plotter.reset)
        self._apply_background_correction_check_box.stateChanged.connect(self._viewer_plotter.reset)
        self._enable_roi_checkbox.stateChanged.connect(self._reset_roi_button.setEnabled)
        self._enable_roi_checkbox.stateChanged.connect(self._save_roi_button.setEnabled)
        self._enable_roi_checkbox.stateChanged.connect(self._clear_roi_button.setEnabled)
        self._enable_roi_checkbox.stateChanged.connect(self._viewer_plotter.enable_region_of_interest)
        self._enable_roi_checkbox.stateChanged.connect(self._change_roi_fields)
        self._viewer_plotter.roi.sigRegionChanged.connect(self._get_roi_values)
        self._reset_roi_button.clicked.connect(self._reset)
        self._save_roi_button.clicked.connect(self._save_roi)
        self._clear_roi_button.clicked.connect(self._viewer_plotter.fit_roi_to_image)

    @property
    def name(self):
        return self._name

    @property
    def start_button(self):
        return self._start_button

    @property
    def stop_button(self):
        return self._stop_button

    @property
    def take_button(self):
        return self._take_button

    @property
    def samples_line_edit(self):
        return self._samples_line_edit

    @property
    def apply_background_correction_check_box(self):
        return self._apply_background_correction_check_box

    def _change_roi_fields(self, state):
        if state:
            self._add_roi_labels()
        else:
            self._remove_roi_labels()

    def _add_roi_labels(self):
        self._roi_x_position_label = QLabel("X:")
        self._roi_x_position_line_edit = self._get_sized_line_edit()
        self._roi_x_position_line_edit.returnPressed.connect(self._set_roi)
        self._roi_y_position_label = QLabel("Y:")
        self._roi_y_position_line_edit = self._get_sized_line_edit()
        self._roi_y_position_line_edit.returnPressed.connect(self._set_roi)
        self._roi_width_label = QLabel("Width:")
        self._roi_width_line_edit = self._get_sized_line_edit()
        self._roi_width_line_edit.returnPressed.connect(self._set_roi)
        self._roi_height_label = QLabel("Height:")
        self._roi_height_line_edit = self._get_sized_line_edit()
        self._roi_height_line_edit.returnPressed.connect(self._set_roi)
        self._get_roi_values()
        self._roi_widgets_layout.addWidget(self._roi_x_position_label)
        self._roi_widgets_layout.addWidget(self._roi_x_position_line_edit)
        self._roi_widgets_layout.addWidget(self._roi_y_position_label)
        self._roi_widgets_layout.addWidget(self._roi_y_position_line_edit)
        self._roi_widgets_layout.addWidget(self._roi_width_label)
        self._roi_widgets_layout.addWidget(self._roi_width_line_edit)
        self._roi_widgets_layout.addWidget(self._roi_height_label)
        self._roi_widgets_layout.addWidget(self._roi_height_line_edit)

    @staticmethod
    def _get_sized_line_edit():
        line_edit = QLineEdit()
        line_edit.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setMaxLength(4)
        line_edit.setMaximumWidth(40)
        line_edit.setValidator(QRegExpValidator(QRegExp("[1-9][0-9]*")))
        return line_edit

    def _remove_roi_labels(self):
        self._roi_x_position_label.deleteLater()
        self._roi_x_position_line_edit.deleteLater()
        self._roi_y_position_label.deleteLater()
        self._roi_y_position_line_edit.deleteLater()
        self._roi_width_label.deleteLater()
        self._roi_width_line_edit.deleteLater()
        self._roi_height_label.deleteLater()
        self._roi_height_line_edit.deleteLater()

    def _reset(self):
        if self._enable_roi_checkbox.isChecked():
            self._reset_roi()
            self._set_roi()

    def _reset_roi(self):
        self._roi_x_position_line_edit.setText(str(self._roi_data.position["x"]))
        self._roi_y_position_line_edit.setText(str(self._roi_data.position["y"]))
        self._roi_width_line_edit.setText(str(self._roi_data.size["width"]))
        self._roi_height_line_edit.setText(str(self._roi_data.size["height"]))

    def _save_roi(self):
        if self._is_input_validation_successful():
            self._roi_data.position["x"] = int(self._roi_x_position_line_edit.text())
            self._roi_data.position["y"] = int(self._roi_y_position_line_edit.text())
            self._roi_data.size["width"] = int(self._roi_width_line_edit.text())
            self._roi_data.size["height"] = int(self._roi_height_line_edit.text())
        self._roi_data.save()
        self._reset()

    def _get_roi_values(self):
        if self._enable_roi_checkbox.isChecked():
            x_position = int(self._viewer_plotter.roi.pos()[0] + round(self._viewer_plotter.roi.size()[0] / 2))
            y_position = int(self._viewer_plotter.roi.pos()[1] + round(self._viewer_plotter.roi.size()[1] / 2))
            width = int(self._viewer_plotter.roi.size()[0])
            height = int(self._viewer_plotter.roi.size()[1])
            self._roi_x_position_line_edit.setText(str(x_position))
            self._roi_y_position_line_edit.setText(str(y_position))
            self._roi_width_line_edit.setText(str(width))
            self._roi_height_line_edit.setText(str(height))

    def _set_roi(self):
        if self._is_input_validation_successful() and self._is_requested_roi_in_bounds():
            self._viewer_plotter.set_roi_position(
                int(int(self._roi_x_position_line_edit.text()) - round(float(self._roi_width_line_edit.text()) / 2)),
                int(int(self._roi_y_position_line_edit.text()) - round(float(self._roi_height_line_edit.text()) / 2)))
            self._viewer_plotter.set_roi_size(int(self._roi_width_line_edit.text()),
                                              int(self._roi_height_line_edit.text()))

    def _is_input_validation_successful(self):
        for line_edit, label in ((self._roi_x_position_line_edit, self._roi_x_position_label),
                                 (self._roi_y_position_line_edit, self._roi_y_position_label),
                                 (self._roi_width_line_edit, self._roi_width_label),
                                 (self._roi_height_line_edit, self._roi_height_label)):
            try:
                int(line_edit.text())
            except ValueError:
                messages.show_error("Empty '" + label.text()[:-1] + "' field")
                return False
        return True

    def _is_requested_roi_in_bounds(self):
        if not self._viewer_plotter.is_requested_roi_in_bounds(x_position=int(self._roi_x_position_line_edit.text()),
                                                               y_position=int(self._roi_y_position_line_edit.text()),
                                                               width=int(self._roi_width_line_edit.text()),
                                                               height=int(self._roi_height_line_edit.text())):
            self._viewer_plotter.fit_roi_to_image()
            messages.show_error("Requested ROI out of bounds")
            return False
        return True

    def show_progress_bar(self):
        self._progress_bar.setValue(0)
        self._take_button_and_progress_bar_stack.setCurrentWidget(self._progress_bar)

    def show_take_button(self):
        self._take_button_and_progress_bar_stack.setCurrentWidget(self._take_button)

    def get_relevant_image_and_offset(self, image):
        if self._enable_roi_checkbox.isChecked():
            return self._viewer_plotter.get_relevant_image_and_offset(image)
        return image, (0, 0)

    def add_to_connections_for_changes(self, action):
        self._viewer_plotter.add_to_connections_for_changes(action)
        self._enable_roi_checkbox.stateChanged.connect(action)


    def plot_image(self, image):
        self._viewer_plotter.set_image(image)
        if not self._image_read_once:
            self._image_read_once = True
            self._enable_roi_checkbox.setEnabled(True)
            self._enable_roi_checkbox.setChecked(True)
            self._reset()

    def plot_projections(self, values, units, range_calibration, invert_x_axis=False):
        self._viewer_plotter.set_projections(values, units, range_calibration, invert_x_axis)

    def plot_fits(self, fits):
        self._viewer_plotter.set_fits(fits)

    def show_fit_parameters(self, fit_parameters, use_encoder, encoder_readout):
        self._viewer_plotter.set_fit_parameters(fit_parameters, use_encoder, encoder_readout)

    def set_progress(self, value):
        self._progress_bar.setValue(value)

    def clear_plotter(self):
        if self._image_read_once:
            self._image_read_once = False
            self._enable_roi_checkbox.setEnabled(False)
            self._enable_roi_checkbox.setChecked(False)
            self._viewer_plotter.clear()

    def activate_roi(self, state):
        if self._image_read_once:
            self._enable_roi_checkbox.setEnabled(state)
            if self._enable_roi_checkbox.isChecked():
                self._reset_roi_button.setEnabled(state)
                self._save_roi_button.setEnabled(state)
                self._clear_roi_button.setEnabled(state)
                self._roi_x_position_line_edit.setEnabled(state)
                self._roi_y_position_line_edit.setEnabled(state)
                self._roi_width_line_edit.setEnabled(state)
                self._roi_height_line_edit.setEnabled(state)
                self._viewer_plotter.activate_roi(state)

    def update_axis(self):
        self._viewer_plotter.update_axis()
