from PyQt5.QtCore import Qt, QRegExp
from PyQt5.QtGui import QRegExpValidator
from PyQt5.QtWidgets import (QMainWindow, QVBoxLayout, QPushButton, QSizePolicy, QCheckBox, QRadioButton, QWidget,
                             QHBoxLayout, QSplitter, QGroupBox, QLineEdit, QLabel)

from ..plotter.logger_plotter import LoggerPlotter


class LoggerToolUi(QMainWindow):
    START_LABEL = "Start"

    _CENTER_LABEL = "Center"
    _FWHM_LABEL = "FWHM"
    _AMPLITUDE_LABEL = "Amplitude"

    def __init__(self):
        super().__init__()
        self._name = "Logger"
        self._main_layout = QVBoxLayout()
        self._logger_plotter = LoggerPlotter(self._CENTER_LABEL, self._FWHM_LABEL, self._AMPLITUDE_LABEL)
        self._start_button = self._get_labeled_push_button(self.START_LABEL)
        self._stop_button = self._get_labeled_push_button("Stop")
        self._reset_button = self._get_labeled_push_button("Reset")
        self._enable_averaging_checkbox = QCheckBox("Enable")
        self._samples_line_edit = self._get_samples_line_edit()
        self._enable_movable_line_checkbox = QCheckBox("Enable", enabled=False)
        self._horizontal_line_label = self._get_sized_line_edit()
        self._vertical_line_label = self._get_sized_line_edit()
        self._horizontal_center_radio_button = QRadioButton(self._CENTER_LABEL, checked=True)
        self._horizontal_fwhm_radio_button = QRadioButton(self._FWHM_LABEL)
        self._horizontal_amplitude_radio_button = QRadioButton(self._AMPLITUDE_LABEL)
        self._vertical_center_radio_button = QRadioButton(self._CENTER_LABEL, checked=True)
        self._vertical_fwhm_radio_button = QRadioButton(self._FWHM_LABEL)
        self._vertical_amplitude_radio_button = QRadioButton(self._AMPLITUDE_LABEL)
        self._create_layout()
        self._create_connections()
        self._read_once = False
        self._horizontal_label = None
        self._vertical_label = None
        self._enable_movable_state = self._logger_plotter.is_movable_line_visible

    @staticmethod
    def _get_labeled_push_button(label):
        push_button = QPushButton(label)
        push_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        return push_button

    @staticmethod
    def _get_samples_line_edit():
        line_edit = QLineEdit("10")
        line_edit.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setValidator(QRegExpValidator(QRegExp("[1-9][0-9]*")))
        line_edit.setMaxLength(3)
        line_edit.setMaximumWidth(33)
        return line_edit

    @staticmethod
    def _get_sized_line_edit():
        line_edit = QLineEdit(enabled=False)
        line_edit.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setValidator(QRegExpValidator(QRegExp("[0-9]*[.]?[0-9]*")))
        line_edit.setMaxLength(6)
        line_edit.setMaximumWidth(55)
        return line_edit

    def _create_layout(self):
        self._create_main_layout()
        self._create_plot_layout()
        self._create_controls_layout()

    def _create_main_layout(self):
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_plot_layout(self):
        plot_layout = QHBoxLayout()
        splitter = QSplitter()
        splitter.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)
        splitter.addWidget(self._logger_plotter.horizontal_widget)
        splitter.addWidget(self._logger_plotter.vertical_widget)
        plot_layout.addWidget(splitter)
        self._main_layout.addLayout(plot_layout)

    def _create_controls_layout(self):
        controls_layout = QHBoxLayout()
        controls_layout.addWidget(self._create_control_group_box())
        controls_layout.addWidget(self._create_averaging_group_box())
        controls_layout.addWidget(self._create_line_marker_group_box())
        controls_layout.addWidget(self._create_horizontal_metrics_group_box())
        controls_layout.addWidget(self._create_vertical_metrics_group_box())
        self._main_layout.addLayout(controls_layout)

    def _create_control_group_box(self):
        group_box = QGroupBox("Control")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._start_button)
        widgets_layout.addWidget(self._stop_button)
        widgets_layout.addWidget(self._reset_button)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_averaging_group_box(self):
        group_box = QGroupBox("Moving Average")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._enable_averaging_checkbox)
        widgets_layout.addWidget(QLabel("Samples:"))
        widgets_layout.addWidget(self._samples_line_edit)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_line_marker_group_box(self):
        group_box = QGroupBox("Line Marker")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._enable_movable_line_checkbox)
        widgets_layout.addWidget(QLabel("Horizontal:"))
        widgets_layout.addWidget(self._horizontal_line_label)
        widgets_layout.addWidget(QLabel("Vertical:"))
        widgets_layout.addWidget(self._vertical_line_label)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_horizontal_metrics_group_box(self):
        group_box = QGroupBox("Horizontal Parameters")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._horizontal_center_radio_button)
        widgets_layout.addWidget(self._horizontal_fwhm_radio_button)
        widgets_layout.addWidget(self._horizontal_amplitude_radio_button)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_vertical_metrics_group_box(self):
        group_box = QGroupBox("Vertical Parameters")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._vertical_center_radio_button)
        widgets_layout.addWidget(self._vertical_fwhm_radio_button)
        widgets_layout.addWidget(self._vertical_amplitude_radio_button)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_connections(self):
        self._enable_movable_line_checkbox.stateChanged.connect(self._clear)
        self._enable_movable_line_checkbox.stateChanged.connect(self._logger_plotter.set_movable_line_visible)
        self._enable_movable_line_checkbox.stateChanged.connect(self._update_labels)
        self._horizontal_center_radio_button.clicked.connect(
            lambda: self._logger_plotter.set_horizontal_movable_line(self._horizontal_center_radio_button.text()))
        self._horizontal_fwhm_radio_button.clicked.connect(
            lambda: self._logger_plotter.set_horizontal_movable_line(self._horizontal_fwhm_radio_button.text()))
        self._horizontal_amplitude_radio_button.clicked.connect(
            lambda: self._logger_plotter.set_horizontal_movable_line(self._horizontal_amplitude_radio_button.text()))
        self._vertical_center_radio_button.clicked.connect(
            lambda: self._logger_plotter.set_vertical_movable_line(self._vertical_center_radio_button.text()))
        self._vertical_fwhm_radio_button.clicked.connect(
            lambda: self._logger_plotter.set_vertical_movable_line(self._vertical_fwhm_radio_button.text()))
        self._vertical_amplitude_radio_button.clicked.connect(
            lambda: self._logger_plotter.set_vertical_movable_line(self._vertical_amplitude_radio_button.text()))
        self._logger_plotter.moved.connect(self._update_labels)
        self._horizontal_line_label.returnPressed.connect(self._set_marker_line)
        self._vertical_line_label.returnPressed.connect(self._set_marker_line)

    @property
    def name(self):
        return self._name

    @property
    def start_button(self):
        return self._start_button

    @property
    def stop_button(self):
        return self._stop_button

    @property
    def reset_button(self):
        return self._reset_button

    @property
    def enable_averaging_checkbox(self):
        return self._enable_averaging_checkbox

    @property
    def samples_line_edit(self):
        return self._samples_line_edit

    @property
    def enable_movable_line_checkbox(self):
        return self._enable_movable_line_checkbox

    def reset(self):
        self._enable_movable_state = self._enable_movable_line_checkbox.isChecked()
        self._logger_plotter.reset()
        self._clear()
        self._read_once = False
        self._enable_movable_line_checkbox.setEnabled(False)
        self._enable_movable_line_checkbox.setChecked(False)

    def _clear(self):
        self._horizontal_line_label.setText("")
        self._vertical_line_label.setText("")

    def enable_labels(self, state):
        self._horizontal_line_label.setEnabled(state)
        self._vertical_line_label.setEnabled(state)

    def add_to_connections_for_changes(self, action):
        self._horizontal_center_radio_button.clicked.connect(action)
        self._horizontal_fwhm_radio_button.clicked.connect(action)
        self._horizontal_amplitude_radio_button.clicked.connect(action)
        self._vertical_center_radio_button.clicked.connect(action)
        self._vertical_fwhm_radio_button.clicked.connect(action)
        self._vertical_amplitude_radio_button.clicked.connect(action)

    def plot_projections(self, values, units):
        self._logger_plotter.set_projections(values, units)

    def plot_fits(self, fits):
        self._logger_plotter.set_fits(fits)

    def plot_metrics(self, values, units):
        self._plot_horizontal_metrics(values.horizontal, units.horizontal)
        self._plot_vertical_metrics(values.vertical, units.vertical)
        self._update_labels()
        if not self._read_once:
            self._read_once = True
            self._enable_movable_line_checkbox.setEnabled(True)
            self._enable_movable_line_checkbox.setChecked(self._enable_movable_state)

    def _plot_horizontal_metrics(self, values, units):
        if self._horizontal_center_radio_button.isChecked():
            self._logger_plotter.set_horizontal_metrics(values.center, units.center,
                                                        self._horizontal_center_radio_button.text())
        elif self._horizontal_fwhm_radio_button.isChecked():
            self._logger_plotter.set_horizontal_metrics(values.fwhm, units.fwhm,
                                                        self._horizontal_fwhm_radio_button.text())
        elif self._horizontal_amplitude_radio_button.isChecked():
            self._logger_plotter.set_horizontal_metrics(values.amplitude, units.amplitude,
                                                        self._horizontal_amplitude_radio_button.text())

    def _plot_vertical_metrics(self, values, units):
        if self._vertical_center_radio_button.isChecked():
            self._logger_plotter.set_vertical_metrics(values.center, units.center,
                                                      self._vertical_center_radio_button.text())
        elif self._vertical_fwhm_radio_button.isChecked():
            self._logger_plotter.set_vertical_metrics(values.fwhm, units.fwhm,
                                                      self._vertical_fwhm_radio_button.text())
        elif self._vertical_amplitude_radio_button.isChecked():
            self._logger_plotter.set_vertical_metrics(values.amplitude, units.amplitude,
                                                      self._vertical_amplitude_radio_button.text())

    def clear_plotter(self):
        if self._read_once:
            self._read_once = False
            self._enable_movable_line_checkbox.setEnabled(False)
            self._enable_movable_line_checkbox.setChecked(False)
            self._logger_plotter.clear()

    def _update_labels(self):
        if self._enable_movable_line_checkbox.isChecked():
            self._update_horizontal_labels()
            self._update_vertical_labels()
            self._logger_plotter.set_labels(self._horizontal_label, self._vertical_label)

    def _update_horizontal_labels(self):
        if self._horizontal_center_radio_button.isChecked():
            self._horizontal_label = self._horizontal_center_radio_button.text()
            label = self._logger_plotter.get_horizontal_label(self._horizontal_label)
            if label is not None:
                self._horizontal_line_label.setText(str(label))
        elif self._horizontal_fwhm_radio_button.isChecked():
            self._horizontal_label = self._horizontal_fwhm_radio_button.text()
            label = self._logger_plotter.get_horizontal_label(self._horizontal_label)
            if label is not None:
                self._horizontal_line_label.setText(str(label))
        elif self._horizontal_amplitude_radio_button.isChecked():
            self._horizontal_label = self._horizontal_amplitude_radio_button.text()
            label = self._logger_plotter.get_horizontal_label(self._horizontal_label)
            if label is not None:
                self._horizontal_line_label.setText(str(label))

    def _update_vertical_labels(self):
        if self._vertical_center_radio_button.isChecked():
            self._vertical_label = self._vertical_center_radio_button.text()
            label = self._logger_plotter.get_vertical_label(self._vertical_label)
            if label is not None:
                self._vertical_line_label.setText(str(label))
        elif self._vertical_fwhm_radio_button.isChecked():
            self._vertical_label = self._vertical_fwhm_radio_button.text()
            label = self._logger_plotter.get_vertical_label(self._vertical_label)
            if label is not None:
                self._vertical_line_label.setText(str(label))
        elif self._vertical_amplitude_radio_button.isChecked():
            self._vertical_label = self._vertical_amplitude_radio_button.text()
            label = self._logger_plotter.get_vertical_label(self._vertical_label)
            if label is not None:
                self._vertical_line_label.setText(str(label))

    def _set_marker_line(self):
        self._logger_plotter.set_position(float(self._horizontal_line_label.text()),
                                          float(self._vertical_line_label.text()))
        self._logger_plotter.set_labels(self._horizontal_label, self._vertical_label)

    def init(self):
        self._logger_plotter.init()
