from PyQt5.QtCore import Qt, QRegExp
from PyQt5.QtGui import QRegExpValidator
from PyQt5.QtWidgets import (QMainWindow, QVBoxLayout, QWidget, QHBoxLayout, QStackedWidget, QPushButton, QSizePolicy,
                             QProgressBar, QLineEdit, QGroupBox, QLabel, QCheckBox, QRadioButton)

from ..plotter.correlator_plotter import CorrelatorPlotter


class CorrelatorToolUi(QMainWindow):
    _GAUSSIAN_LABEL = "Gaussian"
    _FLATTOP_LABEL = "Flat-Top"

    def __init__(self):
        super().__init__()
        self._name = "Correlator"
        self._main_layout = QVBoxLayout()
        self._correlator_plotter = CorrelatorPlotter()
        self._take_button_and_progress_bar_stack = QStackedWidget()
        self._take_button = self._get_labeled_push_button("Take")
        self._progress_bar = QProgressBar()
        self._stop_button = self._get_labeled_push_button("Stop")
        self._samples_line_edit = self._get_samples_line_edit()
        self._automatic_checkbox = QCheckBox("Automatic", checked=True)
        self._fit_range_line_edit = self._get_energy_line_edit()
        self._spectral_bin_line_edit = self._get_energy_line_edit()
        self._subtract_baseline_correction_checkbox = QCheckBox("Subtract Baseline")
        self._process_button = self._get_labeled_push_button("Process")
        self._gaussian_radio_button = QRadioButton(self._GAUSSIAN_LABEL, checked=True)
        self._flattop_radio_button = QRadioButton(self._FLATTOP_LABEL)
        self._create_layout()

    @staticmethod
    def _get_labeled_push_button(label):
        push_button = QPushButton(label)
        push_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        return push_button

    @staticmethod
    def _get_samples_line_edit():
        line_edit = QLineEdit("100")
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setMaxLength(4)
        line_edit.setMaximumWidth(40)
        line_edit.setValidator(QRegExpValidator(QRegExp("[1-9][0-9]*")))
        return line_edit

    @staticmethod
    def _get_energy_line_edit():
        line_edit = QLineEdit()
        line_edit.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setMaxLength(4)
        line_edit.setMaximumWidth(37)
        line_edit.setValidator(QRegExpValidator(QRegExp("[0-9]*[.][0-9]*")))
        return line_edit

    def _create_layout(self):
        self._create_main_layout()
        self._create_plot_layout()
        self._create_controls_layout()

    def _create_main_layout(self):
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_plot_layout(self):
        plot_layout = QHBoxLayout()
        plot_layout.addWidget(self._correlator_plotter.widget)
        self._main_layout.addLayout(plot_layout)

    def _create_controls_layout(self):
        controls_layout = QHBoxLayout()
        controls_layout.addWidget(self._create_control_group_box())
        controls_layout.addWidget(self._create_process_parameters_group_box())
        controls_layout.addWidget(self._create_analysis_group_box())
        controls_layout.addWidget(self._create_g2_fit_group_box())
        self._main_layout.addLayout(controls_layout)

    def _create_control_group_box(self):
        group_box = QGroupBox("Control")
        group_box.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        widgets_layout = QHBoxLayout()
        self._take_button_and_progress_bar_stack.addWidget(self._take_button)
        self._take_button_and_progress_bar_stack.addWidget(self._progress_bar)
        widgets_layout.addWidget(self._take_button_and_progress_bar_stack)
        widgets_layout.addWidget(self._stop_button)
        widgets_layout.addWidget(QLabel("Samples:"))
        widgets_layout.addWidget(self._samples_line_edit)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_process_parameters_group_box(self):
        group_box = QGroupBox("Process Parameters")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._automatic_checkbox)
        widgets_layout.addWidget(QLabel("Fit Range (eV):"))
        widgets_layout.addWidget(self._fit_range_line_edit)
        widgets_layout.addWidget(QLabel("Spectral Bin (eV):"))
        widgets_layout.addWidget(self._spectral_bin_line_edit)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_analysis_group_box(self):
        group_box = QGroupBox("Analysis")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._subtract_baseline_correction_checkbox)
        self._subtract_baseline_correction_checkbox.setEnabled(False)
        widgets_layout.addWidget(self._process_button)
        self._process_button.setEnabled(False)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_g2_fit_group_box(self):
        group_box = QGroupBox("G2 Fit")
        group_box.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._gaussian_radio_button)
        widgets_layout.addWidget(self._flattop_radio_button)
        self.enable_g2_fit_radio_buttons(False)
        group_box.setLayout(widgets_layout)
        return group_box

    @property
    def name(self):
        return self._name

    @property
    def take_button(self):
        return self._take_button

    @property
    def stop_button(self):
        return self._stop_button

    @property
    def samples_line_edit(self):
        return self._samples_line_edit

    @property
    def automatic_checkbox(self):
        return self._automatic_checkbox

    @property
    def fit_range_line_edit(self):
        return self._fit_range_line_edit

    @property
    def spectral_bin_line_edit(self):
        return self._spectral_bin_line_edit

    @property
    def subtract_baseline_correction_checkbox(self):
        return self._subtract_baseline_correction_checkbox

    @property
    def process_button(self):
        return self._process_button

    @property
    def gaussian_radio_button(self):
        return self._gaussian_radio_button

    @property
    def flattop_radio_button(self):
        return self._flattop_radio_button

    def enable_g2_fit_radio_buttons(self, state):
        self._gaussian_radio_button.setEnabled(state)
        self._flattop_radio_button.setEnabled(state)

    def show_progress_bar(self):
        self._progress_bar.setValue(0)
        self._take_button_and_progress_bar_stack.setCurrentWidget(self._progress_bar)

    def show_take_button(self):
        self._take_button_and_progress_bar_stack.setCurrentWidget(self._take_button)

    def set_progress(self, value):
        self._progress_bar.setValue(value)

    def initialize_spectra_plot(self):
        self._correlator_plotter.initialize_spectra_plot()

    def plot_spectra(self, spectra):
        self._correlator_plotter.set_spectra(spectra)

    def initialize_g2_plot(self):
        self._correlator_plotter.initialize_g2_plot()

    def plot_g2(self, delta_energy, g2_data, g2_fit, pulse_duration):
        self._correlator_plotter.set_g2(delta_energy, g2_data, g2_fit, pulse_duration)

    def clear_plotter(self):
        self._correlator_plotter.clear()
