import numpy as np
from doocspie.pyqt import messages

from .correlator_tool_ui import CorrelatorToolUi


class CorrelatorTool:

    def __init__(self, correlation_service, fit_service, calibration_service, metrics_data, roi):
        self._correlator_tool_ui = CorrelatorToolUi()
        self._correlation_service = correlation_service
        self._fit_service = fit_service
        self._calibration_service = calibration_service
        self._correlation_data = self._correlation_service.correlation_data
        self._metrics_data = metrics_data
        self._roi = roi
        self._is_started = None
        self._is_g2_shown = None
        self._create_connections()

    def _create_connections(self):
        self._correlator_tool_ui.take_button.clicked.connect(self._take_button_clicked)
        self._correlator_tool_ui.stop_button.clicked.connect(self.stop_button_clicked)
        self._correlator_tool_ui.automatic_checkbox.toggled.connect(self._conditional_suggest_process_parameters)
        self._correlator_tool_ui.subtract_baseline_correction_checkbox.toggled.connect(
            self._apply_correction_and_suggest_process_parameters)
        self._correlator_tool_ui.process_button.clicked.connect(self._process_button_clicked)
        self._correlator_tool_ui.gaussian_radio_button.toggled.connect(self._show_g2_results)

    @property
    def ui(self):
        return self._correlator_tool_ui

    @property
    def is_started(self):
        return self._is_started

    def _take_button_clicked(self):
        if self._is_control_input_validation_successful():
            self._is_started = True
            self._is_g2_shown = False
            self._correlator_tool_ui.take_button.setEnabled(False)
            self._correlator_tool_ui.samples_line_edit.setEnabled(False)
            self._correlator_tool_ui.show_progress_bar()
            self._correlator_tool_ui.subtract_baseline_correction_checkbox.setEnabled(False)
            self._correlator_tool_ui.process_button.setEnabled(False)
            self._correlator_tool_ui.enable_g2_fit_radio_buttons(False)
            self._correlation_data.initialize(int(self._correlator_tool_ui.samples_line_edit.text()))
            self._correlator_tool_ui.initialize_spectra_plot()

    def set_enabled_take_button(self, state):
        self._correlator_tool_ui.take_button.setEnabled(state)

    def _is_control_input_validation_successful(self):
        try:
            int(self._correlator_tool_ui.samples_line_edit.text())
        except ValueError:
            messages.show_error("Empty 'Samples' field")
            return False
        return True

    def stop_button_clicked(self):
        self._correlator_tool_ui.take_button.setEnabled(True)
        if self._is_started:
            self._correlator_tool_ui.show_take_button()
            self._correlator_tool_ui.samples_line_edit.setEnabled(True)
            self._correlation_data.reset()
            self._is_started = False

    def update_samples_taking(self):
        if self._is_started:
            self._correlator_tool_ui.set_progress(self._correlation_data.get_samples_taking_progress())
            if self._correlation_data.get_samples_taking_progress() == 100:
                self._correlator_tool_ui.subtract_baseline_correction_checkbox.setEnabled(True)
                self._correlator_tool_ui.process_button.setEnabled(True)
                self._correlator_tool_ui.stop_button.click()
                self._set_baseline_fit()
                self._apply_correction_and_suggest_process_parameters()

    def _set_baseline_fit(self):
        baseline_fit = self._fit_service.get_baseline_fit(*self._correlation_data.baseline)
        if baseline_fit is not None:
            self._correlation_data.set_baseline_fit(self._calibration_service.get_energy_calibration(*baseline_fit))
        else:
            self._correlation_data.set_baseline_fit(None)

    def _process_button_clicked(self):
        if self._is_analysis_input_validation_successful():
            self._correlation_service.process(float(self._correlator_tool_ui.fit_range_line_edit.text()),
                                              float(self._correlator_tool_ui.spectral_bin_line_edit.text()))
            self._correlator_tool_ui.enable_g2_fit_radio_buttons(True)
            self._show_g2_results()
            self._is_g2_shown = True

    def _is_analysis_input_validation_successful(self):
        try:
            float(self._correlator_tool_ui.fit_range_line_edit.text())
        except ValueError:
            messages.show_error("Empty/Faulty 'Energy Range' field")
            return False

        try:
            float(self._correlator_tool_ui.spectral_bin_line_edit.text())
        except ValueError:
            messages.show_error("Empty/Faulty 'Energy Bin' field")
            return False

        return True

    def _show_g2_results(self):
        self._correlator_tool_ui.initialize_g2_plot()
        if self._correlator_tool_ui.gaussian_radio_button.isChecked():
            if self._correlation_data.g2_fit_gauss is not None:
                self._correlator_tool_ui.plot_g2(self._correlation_data.delta_energy, self._correlation_data.g2_data,
                                                 self._correlation_data.g2_fit_gauss,
                                                 self._correlation_data.pulse_duration_gauss)
            else:
                messages.show_error("Fit for Gaussian G2 model failed")
        elif self._correlator_tool_ui.flattop_radio_button.isChecked():
            if self._correlation_data.g2_fit_flattop is not None:
                self._correlator_tool_ui.plot_g2(self._correlation_data.delta_energy, self._correlation_data.g2_data,
                                                 self._correlation_data.g2_fit_flattop,
                                                 self._correlation_data.pulse_duration_flattop)
            else:
                messages.show_error("Fit for Flat-Top G2 model failed")

    def update(self):
        if self._roi_size_invalid():
            return
        if self._correlation_data.intensities and not self._is_g2_shown:
            self._correlator_tool_ui.plot_spectra(self._correlation_data.intensities)

    def _roi_size_invalid(self):
        if self._roi.isVisible() and self._roi.size().x() < 3:
            messages.show_error("ROI 'Width' must be larger than 2")
            self._correlator_tool_ui.stop_button.click()
            self.clear_plotter()
            return True
        return False

    def offline_update(self):
        self.update()

    def _apply_correction_and_suggest_process_parameters(self):
        self._correlation_data.initialize_corrections()
        if self._correlator_tool_ui.subtract_baseline_correction_checkbox.isChecked():
            self._correlation_data.subtract_baseline()
        if self._is_g2_shown:
            self._correlator_tool_ui.initialize_spectra_plot()
        self._correlator_tool_ui.plot_spectra(self._correlation_data.intensities)
        self._is_g2_shown = False
        self._correlator_tool_ui.enable_g2_fit_radio_buttons(False)
        if self._correlator_tool_ui.automatic_checkbox.isChecked():
            self._suggest_process_parameters()

    def _suggest_process_parameters(self):
        target_fit_range = None
        for fit_range in np.arange(0, 5, 0.025):
            self._correlation_service.process(fit_range, 0)
            if self._correlation_data.g2_fit_gauss is not None and len(self._correlation_data.g2_fit_gauss) > 10:
                diff = np.diff(np.diff(self._correlation_data.g2_fit_gauss))
                if np.max(diff) > 0:
                    target_fit_range = 2 * self._correlation_data.delta_energy[np.argmax(diff)]
                    break

        if target_fit_range is not None:
            errors = []
            spectral_bins = np.arange(0, 2 * target_fit_range, 0.05)
            for spectral_bin in spectral_bins:
                self._correlation_service.process(target_fit_range, spectral_bin)
                if self._correlation_data.g2_fit_gauss is not None:
                    errors.append(np.sum((self._correlation_data.g2_data - self._correlation_data.g2_fit_gauss) ** 2))

            if errors:
                self._correlator_tool_ui.fit_range_line_edit.setText(str(target_fit_range))
                self._correlator_tool_ui.spectral_bin_line_edit.setText(str(spectral_bins[np.argmin(errors)]))
                return
        self._correlator_tool_ui.fit_range_line_edit.setText("")
        self._correlator_tool_ui.spectral_bin_line_edit.setText("")

    def _conditional_suggest_process_parameters(self):
        if self._correlator_tool_ui.process_button.isEnabled():
            if self._correlator_tool_ui.automatic_checkbox.isChecked():
                self._suggest_process_parameters()

    def clear_plotter(self):
        self._correlator_tool_ui.subtract_baseline_correction_checkbox.setEnabled(False)
        self._correlator_tool_ui.process_button.setEnabled(False)
        self._correlator_tool_ui.enable_g2_fit_radio_buttons(False)
        self._correlator_tool_ui.clear_plotter()
        self._correlation_data.set_empty()
