from PyQt5.QtCore import Qt, QRegExp
from PyQt5.QtGui import QRegExpValidator
from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QLineEdit, QPushButton, QGroupBox, QHBoxLayout, QLabel,
                             QStackedWidget, QWidget)


class CalibrationToolUi(QDialog):

    def __init__(self):
        super().__init__()
        self.setWindowTitle("Calibration Settings")
        self._main_layout = QVBoxLayout()
        self._calibration_menu_widget = QStackedWidget()
        self._fl22_widget = QWidget()
        self._gotthards_widget = QWidget()
        self._calibration_menu_widget.addWidget(self._fl22_widget)
        self._calibration_menu_widget.addWidget(self._gotthards_widget)

        self._a0_line_edit = self._get_float_line_edit()
        self._a1_line_edit = self._get_float_line_edit()
        self._a2_line_edit = self._get_float_line_edit()
        self._a3_line_edit = self._get_float_line_edit()
        self._a4_line_edit = self._get_float_line_edit()
        self._a5_line_edit = self._get_float_line_edit()
        self._x_line_edit = self._get_int_line_edit()
        self._y_line_edit = self._get_int_line_edit()
        self._mm_per_px_line_edit = self._get_float_line_edit()

        self._x_1_line_edit = self._get_absolute_float_line_edit()
        self._x_2_line_edit = self._get_absolute_float_line_edit()

        self._lambda_1_line_edit = self._get_absolute_float_line_edit()
        self._lambda_2_line_edit = self._get_absolute_float_line_edit()

        self._apply_button = self._get_none_default_push_button("Apply")
        self._save_button = self._get_none_default_push_button("Save")
        self._cancel_button = self._get_none_default_push_button("Cancel")
        self._create_layout()

    @staticmethod
    def _get_float_line_edit():
        line_edit = QLineEdit()
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setValidator(QRegExpValidator(QRegExp("[-]?[0-9]*[.]?[0-9]*[e][-]?[0-9][0-9]")))
        return line_edit

    @staticmethod
    def _get_int_line_edit():
        line_edit = QLineEdit()
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setValidator(QRegExpValidator(QRegExp("[0-9]*")))
        return line_edit

    @staticmethod
    def _get_absolute_float_line_edit():
        line_edit = QLineEdit()
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setValidator(QRegExpValidator(QRegExp("[0-9]*[.]?[0-9]*[e][-]?[0-9][0-9]")))
        return line_edit

    @staticmethod
    def _get_none_default_push_button(label):
        push_button = QPushButton(label)
        push_button.setAutoDefault(False)
        return push_button

    def _create_layout(self):
        self.setLayout(self._main_layout)
        self._create_fl22_layout()
        self._create_gotthards_layout()
        self._create_controls_layout()
        self.setModal(True)

    def _create_fl22_layout(self):
        layout = QVBoxLayout()
        self._create_polynomial_coefficients_layout(layout)
        self._create_fit_center_layout(layout)
        self._create_image_layout(layout)
        self._fl22_widget.setLayout(layout)
        self._main_layout.addWidget(self._calibration_menu_widget)

    def _create_polynomial_coefficients_layout(self, layout):
        group_box = QGroupBox("Polynomial Coefficients")
        widgets_layout = QVBoxLayout()
        widgets_layout.addWidget(self._create_polynomial_layout())
        widgets_layout.addWidget(self._create_coefficients_layout())
        group_box.setLayout(widgets_layout)
        layout.addWidget(group_box)

    @staticmethod
    def _create_polynomial_layout():
        group_box = QGroupBox("Polynomial")
        widgets_layout = QHBoxLayout()
        f_x = "f(x) = "
        term_0 = "a<sub>0</sub> "
        term_1 = "+ a<sub>1</sub> x "
        term_2 = "+ a<sub>2</sub> x<sup>2</sup> "
        term_3 = "+ a<sub>3</sub> x<sup>3</sup> "
        term_4 = "+ a<sub>4</sub> x<sup>4</sup> "
        term_5 = "+ a<sub>5</sub> x<sup>5</sup>"
        polynomial = f_x + term_0 + term_1 + term_2 + term_3 + term_4 + term_5
        label = QLabel(polynomial)
        label.setAlignment(Qt.AlignCenter)
        widgets_layout.addWidget(label)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_coefficients_layout(self):
        group_box = QGroupBox("Coefficients")
        widgets_layout = QVBoxLayout()
        widgets_top_layout = QHBoxLayout()
        widgets_top_layout.addWidget(QLabel("a<sub>0</sub>:"))
        widgets_top_layout.addWidget(self._a0_line_edit)
        widgets_top_layout.addWidget(QLabel("a<sub>1</sub>:"))
        widgets_top_layout.addWidget(self._a1_line_edit)
        widgets_middle_layout = QHBoxLayout()
        widgets_middle_layout.addWidget(QLabel("a<sub>2</sub>:"))
        widgets_middle_layout.addWidget(self._a2_line_edit)
        widgets_middle_layout.addWidget(QLabel("a<sub>3</sub>:"))
        widgets_middle_layout.addWidget(self._a3_line_edit)
        widgets_bottom_layout = QHBoxLayout()
        widgets_bottom_layout.addWidget(QLabel("a<sub>4</sub>:"))
        widgets_bottom_layout.addWidget(self._a4_line_edit)
        widgets_bottom_layout.addWidget(QLabel("a<sub>5</sub>:"))
        widgets_bottom_layout.addWidget(self._a5_line_edit)
        widgets_layout.addLayout(widgets_top_layout)
        widgets_layout.addLayout(widgets_middle_layout)
        widgets_layout.addLayout(widgets_bottom_layout)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_fit_center_layout(self, layout):
        group_box = QGroupBox("Fit Center")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(QLabel("x:"))
        widgets_layout.addWidget(self._x_line_edit)
        widgets_layout.addWidget(QLabel("y:"))
        widgets_layout.addWidget(self._y_line_edit)
        group_box.setLayout(widgets_layout)
        layout.addWidget(group_box)

    def _create_image_layout(self, layout):
        group_box = QGroupBox("Image")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(QLabel("millimeter per pixel:"))
        widgets_layout.addWidget(self._mm_per_px_line_edit)
        group_box.setLayout(widgets_layout)
        layout.addWidget(group_box)

    def _create_gotthards_layout(self):
        layout = QVBoxLayout()
        self._create_calibration_points_layout(layout)
        self._gotthards_widget.setLayout(layout)
        self._main_layout.addWidget(self._calibration_menu_widget)

    def _create_calibration_points_layout(self, layout):
        group_box = QGroupBox("Calibration Points")
        widgets_layout = QVBoxLayout()
        widgets_layout.addWidget(self._create_calibration_layout())
        widgets_layout.addWidget(self._create_points_layout())
        group_box.setLayout(widgets_layout)
        layout.addWidget(group_box)

    @staticmethod
    def _create_calibration_layout():
        group_box = QGroupBox("Calibration")
        widgets_layout = QVBoxLayout()
        lambda_x = "\u03bb(x) = "
        calibration_term_0 = "\u03bb<sub>1</sub> "
        calibration_term_1 = "+ D (x-x<sub>1</sub>)"
        calibration = lambda_x + calibration_term_0 + calibration_term_1
        label_top = QLabel(calibration)
        label_top.setAlignment(Qt.AlignCenter)
        widgets_layout.addWidget(label_top)
        dispersion = "D = "
        dispersion_term_0 = "(\u03bb<sub>2</sub> - \u03bb<sub>1</sub>) "
        dispersion_term_1 = "/ (x<sub>2</sub> - x<sub>1</sub>)"
        linear = dispersion + dispersion_term_0 + dispersion_term_1
        label_bottom = QLabel(linear)
        label_bottom.setAlignment(Qt.AlignCenter)
        widgets_layout.addWidget(label_bottom)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_points_layout(self):
        group_box = QGroupBox("Points")
        widgets_layout = QVBoxLayout()
        widgets_top_layout = QHBoxLayout()
        widgets_top_layout.addWidget(QLabel("\u03bb<sub>1</sub> (nm):"))
        widgets_top_layout.addWidget(self._lambda_1_line_edit)
        widgets_top_layout.addWidget(QLabel("x<sub>1</sub> (px):"))
        widgets_top_layout.addWidget(self._x_1_line_edit)
        widgets_bottom_layout = QHBoxLayout()
        widgets_bottom_layout.addWidget(QLabel("\u03bb<sub>2</sub> (nm):"))
        widgets_bottom_layout.addWidget(self._lambda_2_line_edit)
        widgets_bottom_layout.addWidget(QLabel("x<sub>2</sub> (px):"))
        widgets_bottom_layout.addWidget(self._x_2_line_edit)
        widgets_layout.addLayout(widgets_top_layout)
        widgets_layout.addLayout(widgets_bottom_layout)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_controls_layout(self):
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._apply_button)
        widgets_layout.addWidget(self._save_button)
        widgets_layout.addWidget(self._cancel_button)
        self._main_layout.addLayout(widgets_layout)

    @property
    def coefficient_line_edits(self):
        return (self._a0_line_edit, self._a1_line_edit, self._a2_line_edit, self._a3_line_edit, self._a4_line_edit,
                self._a5_line_edit)

    @property
    def fit_center_line_edits(self):
        return self._x_line_edit, self._y_line_edit

    @property
    def mm_per_px_line_edit(self):
        return self._mm_per_px_line_edit

    @property
    def calibration_lambda_line_edits(self):
        return self._lambda_1_line_edit, self._lambda_2_line_edit

    @property
    def calibration_x_line_edits(self):
        return self._x_1_line_edit, self._x_2_line_edit

    @property
    def apply_button(self):
        return self._apply_button

    @property
    def save_button(self):
        return self._save_button

    @property
    def cancel_button(self):
        return self._cancel_button

    def set_fl22_menu(self):
        self._calibration_menu_widget.setCurrentWidget(self._fl22_widget)

    def set_gotthards_menu(self):
        self._calibration_menu_widget.setCurrentWidget(self._gotthards_widget)
