import numpy as np
from PyQt5 import QtCore
from doocspie.util.colors import Colors
from pyqtgraph import GraphicsLayoutWidget, ImageItem, PlotDataItem, LegendItem, HistogramLUTItem, mkPen


class CorrelatorPlotter:
    _DESY_ORANGE = tuple(np.array(Colors.DESY.ORANGE) * 255)
    _DESY_CYAN = np.array(Colors.DESY.CYAN) * 255

    def __init__(self):
        self._widget = GraphicsLayoutWidget()
        self._correlation_plot = self._widget.addPlot()

        self._initialize_design()

        self._spectra = None
        self._histogram = None
        self._g2_data = None
        self._g2_fit = None

        self._legend = None

    def _initialize_design(self):
        self._correlation_plot.setMenuEnabled(False)
        self._correlation_plot.showAxis("top", show=True)
        self._correlation_plot.showAxis("right", show=True)
        for position in "top", "right", "left", "bottom":
            self._correlation_plot.getAxis(position).setStyle(showValues=False, tickLength=0)
        self._correlation_plot.setLabel("bottom", "")
        self._correlation_plot.setLabel("left", "")

    def _initialize_spectra(self):
        self._initialize_design()
        self._correlation_plot.getAxis("bottom").setStyle(showValues=True)
        self._correlation_plot.setLabel("left", "Energy")
        self._correlation_plot.setLabel("bottom", "Sample")

    def _initialize_g2(self):
        self._initialize_design()
        self._g2_data.setData(symbol="o", pen=None, symbolBrush=self._DESY_ORANGE, symbolPen=self._DESY_ORANGE)
        self._g2_fit.setData(pen={"color": self._DESY_CYAN, "width": 3})

    @property
    def widget(self):
        return self._widget

    def initialize_spectra_plot(self):
        self._widget.removeItem(self._correlation_plot)
        if self._histogram is not None:
            self._widget.removeItem(self._histogram)
            self._histogram = None
        self._correlation_plot = self._widget.addPlot()
        self._spectra = ImageItem()
        self._histogram = HistogramLUTItem()
        self._histogram.setImageItem(self._spectra)
        self._widget.addItem(self._histogram)
        self._correlation_plot.addItem(self._spectra)
        self._initialize_spectra()

    def set_spectra(self, spectra):
        spectra = np.array(spectra).T
        self._spectra.setImage(spectra)
        self._histogram.setLevels(spectra.min(), spectra.max())

    def initialize_g2_plot(self):
        self._widget.removeItem(self._correlation_plot)
        if self._histogram is not None:
            self._widget.removeItem(self._histogram)
            self._histogram = None
        self._correlation_plot = self._widget.addPlot()
        self._g2_data = PlotDataItem()
        self._g2_fit = PlotDataItem()
        self._correlation_plot.addItem(self._g2_data)
        self._correlation_plot.addItem(self._g2_fit)
        self._initialize_g2()

    def set_g2(self, delta_energy, g2_data, g2_fit, pulse_duration):
        self._g2_data.setData(x=delta_energy, y=g2_data)
        self._g2_fit.setData(x=delta_energy, y=g2_fit)

        self._legend = LegendItem(offset=(100, 350), pen=mkPen(0.5))
        self._legend.setParentItem(self._correlation_plot)
        self._legend.addItem(self._g2_fit, f"T<sub>FWHM</sub> = {round(pulse_duration)} fs")
        self._legend.setGeometry(QtCore.QRectF(100, 0, self._legend.width() + 10, self._legend.height()))

        self._correlation_plot.getAxis("left").setStyle(showValues=True)
        self._correlation_plot.getAxis("bottom").setStyle(showValues=True)
        self._correlation_plot.setLabel("left", "g2")
        self._correlation_plot.setLabel("bottom", "Delta Energy (eV)")

    def clear(self):
        self.initialize_spectra_plot()
        self._initialize_design()
