import configparser


class CalibrationData:
    _POLYNOMIAL_COEFFICIENTS = "polynomial coefficients"
    _FIT_CENTER = "fit center"
    _IMAGE = "image"
    _CALIBRATION_POINTS = "calibration points"

    def __init__(self, resources_path, cameras):
        self._resources_path = resources_path
        self._cameras = cameras
        self._file = None
        self._config = configparser.ConfigParser()
        self._polynomial_coefficients = None
        self._fit_center = None
        self._image = None
        self._calibration_points = None

    @property
    def polynomial_coefficients(self):
        return self._polynomial_coefficients

    @property
    def fit_center(self):
        return self._fit_center

    @property
    def image(self):
        return self._image

    @property
    def calibration_points(self):
        return self._calibration_points

    def use_camera(self, checked_camera):
        self._file = self._resources_path + self._cameras[checked_camera]["calibration_file"]
        self._update_config()

    def _update_config(self):
        self._config.clear()
        self._config.read(self._file)

        if self._POLYNOMIAL_COEFFICIENTS in self._config:
            self._polynomial_coefficients = {k: float(v) for k, v in self._config.items(self._POLYNOMIAL_COEFFICIENTS)}
        if self._FIT_CENTER in self._config:
            self._fit_center = {k: int(v) for k, v in self._config.items(self._FIT_CENTER)}
        if self._IMAGE in self._config:
            self._image = {k: float(v) for k, v in self._config.items(self._IMAGE)}

        if self._CALIBRATION_POINTS in self._config:
            self._calibration_points = {}
            for k, v in self._config.items(self._CALIBRATION_POINTS):
                self._calibration_points[k] = float(v)

    def save(self):
        if self._POLYNOMIAL_COEFFICIENTS in self._config:
            for key, value in self._polynomial_coefficients.items():
                self._config[self._POLYNOMIAL_COEFFICIENTS][key] = str(value)

        if self._FIT_CENTER in self._config:
            for key, value in self._fit_center.items():
                self._config[self._FIT_CENTER][key] = str(value)

        if self._IMAGE in self._config:
            for key, value in self._image.items():
                self._config[self._IMAGE][key] = str(value)

        if self._CALIBRATION_POINTS in self._config:
            for key, value in self._calibration_points.items():
                self._config[self._CALIBRATION_POINTS][key] = str(value)

        with open(self._file, "w+", encoding="UTF-8") as calibration_file:
            self._config.write(calibration_file)
