from doocspie.pyqt import messages

from .viewer_tool_ui import ViewerToolUi


class ViewerTool:

    def __init__(self, image_data, calibration_service, fit_service, roi_data):
        self._viewer_tool_ui = ViewerToolUi(roi_data)
        self._image_data = image_data
        self._image_data.get_relevant_image_and_offset = self._viewer_tool_ui.get_relevant_image_and_offset
        self._calibration_service = calibration_service
        self._fit_service = fit_service
        self._use_calibration = None
        self._checked_calibration = None
        self._is_background_correction_enabled = False
        self._is_started = None
        self._create_connections()

    def _create_connections(self):
        self._viewer_tool_ui.start_button.clicked.connect(self.start_button_clicked)
        self._viewer_tool_ui.stop_button.clicked.connect(self.stop_button_clicked)
        self._viewer_tool_ui.take_button.clicked.connect(self._take_button_clicked)
        self._viewer_tool_ui.add_to_connections_for_changes(self.offline_update_projections)
        self._viewer_tool_ui.add_to_connections_for_changes(self._image_data.clear_buffer)
        self._viewer_tool_ui.apply_background_correction_check_box.stateChanged.connect(
            self._image_data.set_background_correction)

    @property
    def ui(self):
        return self._viewer_tool_ui

    @property
    def is_started(self):
        return self._is_started

    def use_calibration(self, is_enabled):
        self._use_calibration = is_enabled

    def set_calibration(self, checked_calibration):
        self._checked_calibration = checked_calibration

    def start_button_clicked(self):
        self._is_started = True
        self._viewer_tool_ui.start_button.setText("running ...")
        self._viewer_tool_ui.start_button.setEnabled(False)
        self._viewer_tool_ui.take_button.setEnabled(False)

    def stop_button_clicked(self):
        if self._is_started:
            self._viewer_tool_ui.start_button.setText(self._viewer_tool_ui.START_LABEL)
            self._viewer_tool_ui.start_button.setEnabled(True)
            self._viewer_tool_ui.take_button.setEnabled(True)
            self._viewer_tool_ui.show_take_button()
            self._viewer_tool_ui.samples_line_edit.setEnabled(True)
            self._image_data.reset()
            self._is_started = False

    def _take_button_clicked(self):
        if self._is_input_validation_successful():
            self._is_started = True
            self._viewer_tool_ui.start_button.setEnabled(False)
            self._viewer_tool_ui.samples_line_edit.setEnabled(False)
            self._viewer_tool_ui.show_progress_bar()
            self.set_apply_background_correction_state(False)
            self._image_data.initialize(int(self._viewer_tool_ui.samples_line_edit.text()))

    def _is_input_validation_successful(self):
        try:
            int(self._viewer_tool_ui.samples_line_edit.text())
        except ValueError:
            self.set_apply_background_correction_state(False)
            messages.show_error("Empty 'Samples' field")
            return False
        return True

    def set_apply_background_correction_state(self, state):
        self._is_background_correction_enabled = state
        self._viewer_tool_ui.apply_background_correction_check_box.setChecked(state)
        self._viewer_tool_ui.apply_background_correction_check_box.setEnabled(state)

    def recover_apply_background_correction_enabled_state(self):
        self._viewer_tool_ui.apply_background_correction_check_box.setEnabled(self._is_background_correction_enabled)

    def is_background_correction_enabled(self):
        return self._viewer_tool_ui.apply_background_correction_check_box.isEnabled()

    def update(self):
        self._viewer_tool_ui.plot_image(self._image_data.image)
        self._plot_and_show()

    def _plot_and_show(self):
        if self._use_calibration:
            if self._checked_calibration == self._calibration_service.WAVELENGTH:
                self._viewer_tool_ui.plot_projections(self._image_data.projections_values_calibrated_in_nm_single,
                                                      self._image_data.projections_units_calibrated_in_nm,
                                                      self._calibration_service.get_wavelength_calibration)
                self._viewer_tool_ui.plot_fits(self._fit_service.fit_data.fits_calibrated_in_nm_single)
                self._viewer_tool_ui.show_fit_parameters(self._fit_service.fit_data.parameters_calibrated_in_nm_single,
                                                         self._image_data.use_encoder,
                                                         self._calibration_service.get_encoder_readout())
            elif self._checked_calibration == self._calibration_service.ENERGY:
                self._viewer_tool_ui.plot_projections(self._image_data.projections_values_calibrated_in_ev_single,
                                                      self._image_data.projections_units_calibrated_in_ev,
                                                      self._calibration_service.get_energy_calibration,
                                                      invert_x_axis=True)
                self._viewer_tool_ui.plot_fits(self._fit_service.fit_data.fits_calibrated_in_ev_single)
                self._viewer_tool_ui.show_fit_parameters(self._fit_service.fit_data.parameters_calibrated_in_ev_single,
                                                         self._image_data.use_encoder,
                                                         self._calibration_service.get_encoder_readout())
        else:
            self._viewer_tool_ui.plot_projections(self._image_data.projections_values_single,
                                                  self._image_data.projections_units,
                                                  self._calibration_service.get_pixel_calibration)
            self._viewer_tool_ui.plot_fits(self._fit_service.fit_data.fits_single)
            self._viewer_tool_ui.show_fit_parameters(self._fit_service.fit_data.parameters_single,
                                                     self._image_data.use_encoder,
                                                     self._calibration_service.get_encoder_readout())

    def update_background_taking(self):
        if self._image_data.to_take_background:
            self._viewer_tool_ui.set_progress(self._image_data.get_background_taking_progress())
            if self._image_data.get_background_taking_progress() == 100:
                self.set_apply_background_correction_state(True)
                self._viewer_tool_ui.stop_button.click()

    def offline_update(self):
        if self._image_data.is_not_empty():
            self.update()

    def offline_update_projections(self):
        if not self._is_started and self._image_data.is_not_empty():
            self._image_data.update_projections()
            self._calibration_service.update()
            self._fit_service.update()
            self._plot_and_show()
            self._viewer_tool_ui.update_axis()

    def clear_plotter(self):
        self._viewer_tool_ui.clear_plotter()
        self._image_data.set_empty()
