from doocspie.pyqt import ELogPrinter, Timer, messages

from .calibration_tool import CalibrationTool
from .correlator_tool import CorrelatorTool
from .logger_tool import LoggerTool
from .spectralyzer_tool_ui import SpectralyzerToolUi
from .viewer_tool import ViewerTool


class SpectralyzerTool:

    def __init__(self, application_name, version, elogs, io_service, calibration_service, fit_service, metrics_service,
                 correlation_service, roi_data):
        self._viewer_tool = ViewerTool(io_service.image_data, calibration_service, fit_service, roi_data)
        self._logger_tool = LoggerTool(io_service.image_data, fit_service.fit_data, metrics_service.metrics_data,
                                       calibration_service)
        self._calibration_tool = CalibrationTool(calibration_service)
        self._correlator_tool = CorrelatorTool(correlation_service, fit_service, calibration_service,
                                               metrics_service.metrics_data, self._viewer_tool.ui.roi)
        self._spectralyzer_tool_ui = SpectralyzerToolUi(application_name, version, io_service.cameras,
                                                        calibration_service.menu_items,
                                                        self._viewer_tool.ui, self._logger_tool.ui,
                                                        self._correlator_tool.ui)
        self._e_log_printer = ELogPrinter(self._spectralyzer_tool_ui, printers=elogs, print_exceptions=True)
        self._concurrency_service = Timer(self._process_loop)
        self._io_service = io_service
        self._calibration_service = calibration_service
        self._fit_service = fit_service
        self._metrics_service = metrics_service
        self._correlation_service = correlation_service
        self._roi_data = roi_data
        self._selected_camera = None
        self._viewer_tool_updated_once = False
        self._update_tools_with_calibration_option()
        self._create_connections()

    def _update_tools_with_calibration_option(self):
        self._spectralyzer_tool_ui.calibration_menu_action_group.setEnabled(
            self._spectralyzer_tool_ui.enable_calibration_action.isChecked())
        self._viewer_tool.use_calibration(self._spectralyzer_tool_ui.enable_calibration_action.isChecked())
        self._viewer_tool.set_calibration(self._spectralyzer_tool_ui.get_checked_calibration())
        self._logger_tool.use_calibration(self._spectralyzer_tool_ui.enable_calibration_action.isChecked())
        self._logger_tool.set_calibration(self._spectralyzer_tool_ui.get_checked_calibration())
        self._update_selected_camera()

    def _update_selected_camera(self):
        if self._spectralyzer_tool_ui.get_checked_camera() != self._selected_camera:
            self._viewer_tool.clear_plotter()
            self._logger_tool.clear_plotter()
            self._logger_tool.ui.init()
            self._correlator_tool.clear_plotter()
            self._selected_camera = self._spectralyzer_tool_ui.get_checked_camera()
            self._io_service.use_camera(self._selected_camera)
            self._calibration_service.use_camera(self._selected_camera)
            self._roi_data.use_camera(self._selected_camera)
            if self._viewer_tool.is_background_correction_enabled():
                self._viewer_tool.set_apply_background_correction_state(False)
                messages.show_warning("No compatible background stored")
            self._viewer_tool_updated_once = False

    def _create_connections(self):
        self._create_spectralyzer_tool_connections()
        self._create_calibration_tool_connections()
        self._create_viewer_tool_connections()
        self._create_logger_tool_connections()
        self._create_correlator_tool_connections()

    def _create_spectralyzer_tool_connections(self):
        self._spectralyzer_tool_ui.print_to_e_log_action.triggered.connect(self._e_log_printer.show_dialog)
        self._spectralyzer_tool_ui.camera_menu_action_group.triggered.connect(self._update_selected_camera)
        self._spectralyzer_tool_ui.enable_calibration_action.triggered.connect(self._enable_calibration)
        self._spectralyzer_tool_ui.enable_calibration_action.triggered.connect(self._logger_tool.ui.init)
        self._spectralyzer_tool_ui.change_calibration_action.triggered.connect(
            self._calibration_tool.change_calibration)
        self._spectralyzer_tool_ui.calibration_menu_action_group.triggered.connect(self._logger_tool.ui.init)
        self._spectralyzer_tool_ui.calibration_menu_action_group.triggered.connect(self._set_calibration)
        self._spectralyzer_tool_ui.tab_widget.currentChanged.connect(self._offline_update_tools)

    def _enable_calibration(self):
        self._spectralyzer_tool_ui.calibration_menu_action_group.setEnabled(
            self._spectralyzer_tool_ui.enable_calibration_action.isChecked())
        self._viewer_tool.use_calibration(self._spectralyzer_tool_ui.enable_calibration_action.isChecked())
        self._logger_tool.use_calibration(self._spectralyzer_tool_ui.enable_calibration_action.isChecked())
        self._offline_update_tools()

    def _set_calibration(self):
        self._viewer_tool.set_calibration(self._spectralyzer_tool_ui.get_checked_calibration())
        self._logger_tool.set_calibration(self._spectralyzer_tool_ui.get_checked_calibration())
        self._offline_update_tools()

    def _create_calibration_tool_connections(self):
        self._calibration_tool.changed.connect(self._viewer_tool.offline_update_projections)

    def _create_viewer_tool_connections(self):
        self._create_tool_connections(caller=self._viewer_tool, callee=self._logger_tool)
        self._viewer_tool.ui.take_button.clicked.connect(self._start_background_taking_process_loop)

    def _create_logger_tool_connections(self):
        self._create_tool_connections(caller=self._logger_tool, callee=self._viewer_tool)

    def _create_tool_connections(self, caller, callee):
        caller.ui.start_button.clicked.connect(self._start_process_loop)
        caller.ui.start_button.clicked.connect(callee.start_button_clicked)
        caller.ui.stop_button.clicked.connect(self._stop_process_loop)
        caller.ui.stop_button.clicked.connect(callee.stop_button_clicked)
        caller.ui.stop_button.clicked.connect(self._correlator_tool.stop_button_clicked)

    def _create_correlator_tool_connections(self):
        self._correlator_tool.ui.take_button.clicked.connect(self._start_correlation_taking_process_loop)
        self._correlator_tool.ui.stop_button.clicked.connect(self._stop_process_loop)
        self._correlator_tool.ui.stop_button.clicked.connect(self._viewer_tool.stop_button_clicked)
        self._correlator_tool.ui.stop_button.clicked.connect(self._logger_tool.stop_button_clicked)

    def _start_process_loop(self):
        if not self._concurrency_service.is_active():
            self._spectralyzer_tool_ui.camera_menu.setEnabled(False)
            self._concurrency_service.start()

    def _start_background_taking_process_loop(self):
        if self._viewer_tool.is_started:
            self._start_process_loop()
            self._logger_tool.start_button_clicked()
            self._correlator_tool.set_enabled_take_button(False)

    def _start_correlation_taking_process_loop(self):
        if self._correlator_tool.is_started:
            self._start_process_loop()
            self._viewer_tool.start_button_clicked()
            self._logger_tool.start_button_clicked()
            self._viewer_tool.ui.activate_roi(False)
            self._viewer_tool.ui.apply_background_correction_check_box.setEnabled(False)
            self._logger_tool.switch_off_averaging()

    def _stop_process_loop(self):
        if self._concurrency_service.is_active():
            self._viewer_tool.ui.activate_roi(True)
            self._viewer_tool.recover_apply_background_correction_enabled_state()
            self._logger_tool.recover_averaging()
            self._spectralyzer_tool_ui.camera_menu.setEnabled(True)
            self._concurrency_service.stop()

    def _process_loop(self):
        self._update_services()
        self._update_tools()

    def _update_services(self):
        self._io_service_update()
        if self._io_service.is_new_data_available():
            self._calibration_service.update()
            self._fit_service.update()
            self._metrics_service.update()
            if self._viewer_tool_updated_once:
                self._correlation_service_update()

    def _io_service_update(self):
        try:
            self._io_service.update()
        except (self._io_service.IOException,
                self._io_service.image_data.ImageSizeChangedWhileTakingBackground,
                self._io_service.image_data.ImageSizeChangedWhileRunning,
                self._io_service.image_data.NoCameraImageType) as exc:
            self._viewer_tool.ui.stop_button.click()
            self._logger_tool.ui.stop_button.click()
            if isinstance(exc, self._io_service.image_data.ImageSizeChangedWhileRunning):
                self._viewer_tool.set_apply_background_correction_state(False)
            messages.show_error(str(exc))

    def _correlation_service_update(self):
        try:
            self._correlation_service.update()
        except self._correlation_service.correlation_data.ImageSizeChangedWhileTakingCorrelationData as exc:
            self._correlator_tool.ui.stop_button.click()
            messages.show_error(str(exc))

    def _update_tools(self):
        if self._io_service.is_new_data_available():
            if self._spectralyzer_tool_ui.get_current_ui() == self._viewer_tool.ui:
                self._viewer_tool.update()
                self._viewer_tool_updated_once = True
            elif self._spectralyzer_tool_ui.get_current_ui() == self._logger_tool.ui:
                self._logger_tool.update()
            elif self._spectralyzer_tool_ui.get_current_ui() == self._correlator_tool.ui:
                self._correlator_tool.update()

            if not self._viewer_tool_updated_once:
                self._viewer_tool.update()
                self._viewer_tool_updated_once = True
                if self._correlator_tool.is_started:
                    self._viewer_tool.ui.activate_roi(False)

            self._viewer_tool.update_background_taking()
            self._correlator_tool.update_samples_taking()

    def _offline_update_tools(self):
        if not self._concurrency_service.is_active():
            if self._spectralyzer_tool_ui.get_current_ui() == self._viewer_tool.ui:
                self._viewer_tool.offline_update()
            elif self._spectralyzer_tool_ui.get_current_ui() == self._logger_tool.ui:
                self._logger_tool.offline_update()
            elif self._spectralyzer_tool_ui.get_current_ui() == self._correlator_tool.ui:
                self._correlator_tool.offline_update()

    def show_window(self):
        self._spectralyzer_tool_ui.show_window()
