from doocspie.pyqt import messages

from .logger_tool_ui import LoggerToolUi


class LoggerTool:

    def __init__(self, image_data, fit_data, metrics_data, calibration_service):
        self._logger_tool_ui = LoggerToolUi()
        self._image_data = image_data
        self._fit_data = fit_data
        self._calibration_service = calibration_service
        self._metrics_data = metrics_data
        self._use_calibration = None
        self._checked_calibration = None
        self._is_started = False
        self._is_averaging_selected = self._logger_tool_ui.enable_averaging_checkbox.isChecked()
        self._create_connections()

    def _create_connections(self):
        self._logger_tool_ui.start_button.clicked.connect(self.start_button_clicked)
        self._logger_tool_ui.stop_button.clicked.connect(self.stop_button_clicked)
        self._logger_tool_ui.reset_button.clicked.connect(self._reset_button_clicked)
        self._logger_tool_ui.add_to_connections_for_changes(self._offline_update_metrics)
        self._logger_tool_ui.enable_averaging_checkbox.toggled.connect(self._set_averaging)
        self._logger_tool_ui.enable_movable_line_checkbox.stateChanged.connect(self._enable_labels)
        self._logger_tool_ui.samples_line_edit.editingFinished.connect(self._set_samples)

    @property
    def ui(self):
        return self._logger_tool_ui

    def _enable_labels(self, state):
        if not self._is_started:
            self._logger_tool_ui.enable_labels(state)

    def use_calibration(self, is_enabled):
        self._use_calibration = is_enabled

    def set_calibration(self, checked_calibration):
        self._checked_calibration = checked_calibration

    def start_button_clicked(self):
        if self._is_input_validation_successful():
            self._is_started = True
            self._logger_tool_ui.start_button.setText("running ...")
            self._logger_tool_ui.start_button.setEnabled(False)
            self._logger_tool_ui.enable_labels(False)

    def stop_button_clicked(self):
        if self._is_started:
            self._logger_tool_ui.start_button.setText(self._logger_tool_ui.START_LABEL)
            self._logger_tool_ui.start_button.setEnabled(True)
            if self._logger_tool_ui.enable_movable_line_checkbox.isChecked():
                self._logger_tool_ui.enable_labels(True)
            self._is_started = False

    def _reset_button_clicked(self):
        if self._image_data.is_not_empty():
            self._metrics_data.reset()
            self._offline_update_metrics()
            self._logger_tool_ui.reset()
            self._image_data.set_empty()

    def update(self):
        if self._use_calibration:
            if self._checked_calibration == self._calibration_service.WAVELENGTH:
                self._logger_tool_ui.plot_projections(self._image_data.projections_values_calibrated_in_nm,
                                                      self._image_data.projections_units_calibrated_in_nm)
                self._logger_tool_ui.plot_fits(self._fit_data.fits_calibrated_in_nm)
                self._logger_tool_ui.plot_metrics(self._metrics_data.metrics_values_calibrated_in_nm,
                                                  self._metrics_data.metrics_units_calibrated_in_nm)
            elif self._checked_calibration == self._calibration_service.ENERGY:
                self._logger_tool_ui.plot_projections(self._image_data.projections_values_calibrated_in_ev,
                                                      self._image_data.projections_units_calibrated_in_ev)
                self._logger_tool_ui.plot_fits(self._fit_data.fits_calibrated_in_ev)
                self._logger_tool_ui.plot_metrics(self._metrics_data.metrics_values_calibrated_in_ev,
                                                  self._metrics_data.metrics_units_calibrated_in_ev)
        else:
            self._logger_tool_ui.plot_projections(self._image_data.projections_values,
                                                  self._image_data.projections_units)
            self._logger_tool_ui.plot_fits(self._fit_data.fits)
            self._logger_tool_ui.plot_metrics(self._metrics_data.metrics_values, self._metrics_data.metrics_units)

    def offline_update(self):
        if self._image_data.is_not_empty():
            self.update()

    def _offline_update_metrics(self):
        if not self._is_started and self._image_data.is_not_empty():
            if self._use_calibration:
                if self._checked_calibration == self._calibration_service.WAVELENGTH:
                    self._logger_tool_ui.plot_metrics(self._metrics_data.metrics_values_calibrated_in_nm,
                                                      self._metrics_data.metrics_units_calibrated_in_nm)
                elif self._checked_calibration == self._calibration_service.ENERGY:
                    self._logger_tool_ui.plot_metrics(self._metrics_data.metrics_values_calibrated_in_ev,
                                                      self._metrics_data.metrics_units_calibrated_in_ev)
            else:
                self._logger_tool_ui.plot_metrics(self._metrics_data.metrics_values,
                                                  self._metrics_data.metrics_units)

    def clear_plotter(self):
        self._metrics_data.reset()
        self._logger_tool_ui.clear_plotter()
        self._logger_tool_ui.reset()

    def _set_averaging(self, state):
        if state and self._is_input_validation_successful():
            self._image_data.set_averaging(True)
            self._image_data.set_samples(int(self._logger_tool_ui.samples_line_edit.text()))
        else:
            self._image_data.set_averaging(False)

    def _is_input_validation_successful(self):
        if self._logger_tool_ui.enable_averaging_checkbox.isChecked():
            try:
                int(self._logger_tool_ui.samples_line_edit.text())
            except ValueError:
                self._logger_tool_ui.enable_averaging_checkbox.setChecked(False)
                messages.show_error("Empty 'Samples' field")
                return False
        return True

    def _set_samples(self):
        samples = int(self._logger_tool_ui.samples_line_edit.text())
        if samples != self._image_data.samples:
            self._image_data.set_samples(samples)

    def switch_off_averaging(self):
        self._logger_tool_ui.enable_averaging_checkbox.setEnabled(False)
        is_averaging_selected = self._logger_tool_ui.enable_averaging_checkbox.isChecked()
        self._logger_tool_ui.enable_averaging_checkbox.setChecked(False)
        self._is_averaging_selected = is_averaging_selected  # using the state before setting the checkbox to false

    def recover_averaging(self):
        if not self._logger_tool_ui.enable_averaging_checkbox.isEnabled():
            self._logger_tool_ui.enable_averaging_checkbox.setEnabled(True)
            self._logger_tool_ui.enable_averaging_checkbox.setChecked(self._is_averaging_selected)
