from PyQt5.QtCore import pyqtSignal, QObject, QSize

from doocspie.pyqt import messages
from .calibration_tool_ui import CalibrationToolUi


class CalibrationTool(QObject):
    changed = pyqtSignal()

    def __init__(self, calibration_service):
        super().__init__()
        self._calibration_tool_ui = CalibrationToolUi()
        self._calibration_service = calibration_service
        self._create_connections()

    def _create_connections(self):
        self._calibration_tool_ui.apply_button.clicked.connect(self._apply_button_clicked)
        self._calibration_tool_ui.save_button.clicked.connect(self._save_button_clicked)
        self._calibration_tool_ui.cancel_button.clicked.connect(self._calibration_tool_ui.close)

    @property
    def ui(self):
        return self._calibration_tool_ui

    def change_calibration(self):
        self._initialize_calibration_tool_ui()
        self._calibration_tool_ui.show()
        self._calibration_tool_ui.exec()

    def _initialize_calibration_tool_ui(self):
        if self._calibration_service.camera == self._calibration_service.SPECTROMETER_FL22:
            self._initialize_f22_calibration_tool_ui()
            self._calibration_tool_ui.set_fl22_menu()
            self._calibration_tool_ui.setFixedSize(QSize(375, 450))

        elif self._calibration_service.camera in (self._calibration_service.GOTTHARD,
                                                  self._calibration_service.GOTTHARD2):
            self._initialize_gotthards_calibration_tool_ui()
            self._calibration_tool_ui.set_gotthards_menu()
            self._calibration_tool_ui.setFixedSize(QSize(375, 300))

    def _initialize_f22_calibration_tool_ui(self):
        for index, line_edit in enumerate(self._calibration_tool_ui.coefficient_line_edits):
            line_edit.setText(self._calibration_service.get_coefficient(index))

        for index, line_edit in enumerate(self._calibration_tool_ui.fit_center_line_edits):
            line_edit.setText(self._calibration_service.get_fit_center(index))

        self._calibration_tool_ui.mm_per_px_line_edit.setText(self._calibration_service.get_image_mm_per_px())

    def _initialize_gotthards_calibration_tool_ui(self):
        for index, line_edit in enumerate(self._calibration_tool_ui.calibration_lambda_line_edits, 1):
            line_edit.setText(self._calibration_service.get_calibration_lambda(index))

        for index, line_edit in enumerate(self._calibration_tool_ui.calibration_x_line_edits, 1):
            line_edit.setText(self._calibration_service.get_calibration_x(index))

    def _apply_button_clicked(self):
        if self._is_input_validation_successful():
            self._set_calibration()
            self._calibration_tool_ui.close()

    def _save_button_clicked(self):
        if self._is_input_validation_successful():
            self._set_calibration()
            self._calibration_service.save_calibration()
            self._calibration_tool_ui.close()

    def _is_input_validation_successful(self):
        if self._calibration_service.camera == self._calibration_service.SPECTROMETER_FL22:
            return self._fl22_input_validation()
        elif self._calibration_service.camera in (self._calibration_service.GOTTHARD,
                                                  self._calibration_service.GOTTHARD2):
            return self._gotthards_input_validation()

    def _fl22_input_validation(self):
        for index, line_edit in enumerate(self._calibration_tool_ui.coefficient_line_edits):
            try:
                float(line_edit.text())
            except ValueError:
                messages.show_error("Empty/Faulty 'a<sub>" + str(index) + "</sub>' field")
                return False

        for index, line_edit in enumerate(self._calibration_tool_ui.fit_center_line_edits):
            try:
                int(line_edit.text())
            except ValueError:
                messages.show_error("Empty '" + ("y" if index else "x") + "' field")
                return False

        try:
            float(self._calibration_tool_ui.mm_per_px_line_edit.text())
        except ValueError:
            messages.show_error("Empty/Faulty 'millimeter per pixel' field")
            return False

        return True

    def _gotthards_input_validation(self):
        for index, line_edit in enumerate(self._calibration_tool_ui.calibration_lambda_line_edits, 1):
            try:
                float(line_edit.text())
            except ValueError:
                messages.show_error("Empty '\u03bb<sub>" + str(index) + "</sub>' field")
                return False

        for index, line_edit in enumerate(self._calibration_tool_ui.calibration_x_line_edits, 1):
            try:
                float(line_edit.text())
            except ValueError:
                messages.show_error("Empty 'x<sub>" + str(index) + "</sub>' field")
                return False

        if (self._calibration_tool_ui.calibration_x_line_edits[0].text() ==
                self._calibration_tool_ui.calibration_x_line_edits[1].text()):
            messages.show_error("'x<sub>1</sub>' and 'x<sub>2</sub>' cannot be equal")
            return False

        return True

    def _set_calibration(self):
        if self._calibration_service.camera == self._calibration_service.SPECTROMETER_FL22:
            self._set_fl22_calibration()
        elif self._calibration_service.camera in (self._calibration_service.GOTTHARD,
                                                  self._calibration_service.GOTTHARD2):
            self._set_gotthards_calibration()

    def _set_fl22_calibration(self):
        for index, line_edit in enumerate(self._calibration_tool_ui.coefficient_line_edits):
            self._calibration_service.set_coefficient(index, float(line_edit.text()))
        self._calibration_service.update_polynomial()

        for index, line_edit in enumerate(self._calibration_tool_ui.fit_center_line_edits):
            self._calibration_service.set_fit_center(index, int(line_edit.text()))

        self._calibration_service.set_image_mm_per_px(float(self._calibration_tool_ui.mm_per_px_line_edit.text()))

        self.changed.emit()

    def _set_gotthards_calibration(self):
        for index, line_edit in enumerate(self._calibration_tool_ui.calibration_lambda_line_edits, 1):
            self._calibration_service.set_calibration_lambda(index, float(line_edit.text()))

        for index, line_edit in enumerate(self._calibration_tool_ui.calibration_x_line_edits, 1):
            self._calibration_service.set_calibration_x(index, float(line_edit.text()))

        self._calibration_service.update_calibration()

        self.changed.emit()
