import os
import signal
import sys

from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QApplication

from .materials.calibration_data import CalibrationData
from .materials.correlation_data import CorrelationData
from .materials.encoder_data import EncoderData
from .materials.fit_data import FitData
from .materials.image_data import ImageData
from .materials.metrics_data import MetricsData
from .materials.roi_data import ROIData
from .services.calibration_service import CalibrationService
from .services.correlation_service import CorrelationService
from .services.fit_service import FitService
from .services.io_service import IOService
from .services.metrics_service import MetricsService
from .tools.spectralyzer_tool import SpectralyzerTool

__application_name__ = "Spectralyzer"
__version__ = "0.3.1"
__elogs__ = {"FLASH": "ttflog", "FLASH2 Photon Diag.": "fl2_photlog", "FLASH2 FL23": "fl2_fl23log",
             "FLASH2 FL24": "fl2_fl24log", "FLASH2 FL26": "fl2_fl26log"}

__cameras__ = {"Spectrometer@FL22": {"address": "FLASH.FEL/FL22.CAM1/PCO.CAM1/IMAGE_EXT_ZMQ",
                                     "calibration_file": "calibration_fl22.ini",
                                     "roi_file": "roi_fl22.ini", "encoder": True},
               "Gotthard": {"address": "FLASH.FEL/GOTTHARD/FL2/IMAGE.NO.BKGR",
                            "calibration_file": "calibration_gotthard.ini",
                            "roi_file": "roi_gotthard.ini", "encoder": False},
               "Gotthard2": {"address": "FLASH.FEL/GOTTHARD2/FL2/IMAGE.NO.BKGR",
                             "calibration_file": "calibration_gotthard2.ini",
                             "roi_file": "roi_gotthard2.ini", "encoder": False}
               }


def startup_application():
    signal.signal(signal.SIGINT, signal.SIG_DFL)  # handles Ctrl-C interrupts via default signal (core dump and exit)

    resources_path = os.path.dirname(os.path.abspath(__file__)) + "/resources/"

    application = QApplication([__application_name__])
    application.setWindowIcon(QIcon(resources_path + "spectralyzer.png"))

    image_data = ImageData(__cameras__)
    encoder_data = EncoderData(address="FLASH.FEL/FL21.MOTOR/MOTOR7/FODOMETER")
    calibration_data = CalibrationData(resources_path, __cameras__)
    correlation_data = CorrelationData()
    fit_data = FitData()
    metrics_data = MetricsData()
    roi_data = ROIData(resources_path, __cameras__)

    io_service = IOService(image_data, encoder_data)
    calibration_service = CalibrationService(calibration_data, encoder_data, image_data)
    fit_service = FitService(image_data, fit_data)
    metrics_service = MetricsService(metrics_data, fit_data)
    correlation_service = CorrelationService(image_data, correlation_data)

    spectralyzer_tool = SpectralyzerTool(__application_name__, __version__, __elogs__,
                                         io_service, calibration_service, fit_service, metrics_service,
                                         correlation_service, roi_data)
    spectralyzer_tool.show_window()

    sys.exit(application.exec())
