"""Utility functions for convenient working with doocspie.

This module provides the utility functions for convenient working with the doocspie output.
"""
import io

import PIL.Image
import numpy as np


def get_time_axis(readout):
    """Get the time axis for a readout object of a given spectral DOOCS property.

    Args:
        readout (Readout): The readout object of spectral DOOCS properties.

    Returns:
        numpy.ndarray: The time axis of the spectral DOOCS property.

    Raises:
        TypeError: Exception for wrong 'readout.type' type.
    """
    if readout.type not in ("SPECTRUM", "GSPECTRUM"):
        raise TypeError("readout.type must be of type 'SPECTRUM' or 'GSPECTRUM'")

    start = readout.meta.start
    increment = readout.meta.incr
    elements = len(readout.data)

    if readout.type == "SPECTRUM" or (readout.type == "GSPECTRUM" and readout.meta.groups <= 1):
        return np.linspace(start, start + (elements - 1) * increment, elements)

    time_axis = np.zeros(elements)
    group_size = readout.meta.groupsize
    group_increment = readout.meta.groupincr
    for index in range(elements):
        group_index = index // group_size  # force integer division
        time_axis[index] = start + index * increment + group_index * group_increment * increment
    return time_axis


def get_image(readout, swap_bytes=False):
    """Get the image for a readout object of a given image-like DOOCS byte array property.

    Args:
        readout (Readout): The readout object of image-like DOOCS byte arrays.
        swap_bytes (bool, optional): The optional state for swapping bytes.

    Returns:
        numpy.ndarray: The image of the image-like DOOCS byte array property.

    Raises:
        TypeError: Exception for wrong 'readout.type' type.
        ValueError: Exception for wrong 'readout.meta.option' option.
    """
    if not readout.type == "A_BYTE":
        raise TypeError("readout.type must be of type 'A_BYTE'")

    if readout.meta.option & 0x1:
        return np.asarray(PIL.Image.open(io.BytesIO(readout.data)))
    if readout.meta.option & 0x2:
        return readout.data.reshape(readout.meta.y_dim, readout.meta.x_dim)
    if readout.meta.option & 0x4:
        image = readout.data.reshape(readout.meta.y_dim, readout.meta.x_dim)
        if swap_bytes:
            return image[:, 0::2] + 2 ** 8 * image[:, 1::2].astype(np.uint16)
        return 2 ** 8 * image[:, 0::2].astype(np.uint16) + image[:, 1::2]
    raise ValueError("option " + str(readout.meta.option) + " is not implemented")
