"""Module with export service class for exporting data to files.

This module provides the export service class with a method for exporting data to files.
"""
from datetime import datetime

import pandas as pd
from PyQt5.QtWidgets import QFileDialog


class ExportService:
    """Export service class for exporting data to files.

    This class provides the method for exporting data to files.
    """

    def __init__(self, application_name, export_directory):
        """Constructor of the export service class.

        This constructor initializes the export service class using the directory to export data to.

        Args:
            application_name (str): The name of the application being part of the file name.
            export_directory (str): The directory to export data to.
        """
        self._application_name = application_name
        self._export_directory = export_directory
        self._options = QFileDialog.Options() | QFileDialog.DontUseNativeDialog

    def export(self, file_name, content):
        """Exporting the given data content to a given file name.

        Args:
            file_name (str): The file name to export data to.
            content (dict): The data content to export.

        Returns:
            None
        """
        full_file_name = self._export_directory + self._application_name + "_" + file_name + "_"
        full_file_name += str(datetime.now().isoformat("T", "seconds")) + ".csv"
        file_name, _ = QFileDialog.getSaveFileName(None, "Export to File", full_file_name, options=self._options)
        if file_name:
            data_frame = pd.DataFrame.from_dict(content, orient="index").T
            data_frame.to_csv(file_name, index=False)
