"""Module with print tool ui class for providing print dialogs.

This module provides the print tool ui class for providing dialogs for printing.
"""
import sys

from PyQt5.QtWidgets import (QDialog, QLineEdit, QComboBox, QTextEdit, QPushButton, QVBoxLayout, QHBoxLayout,
                             QSizePolicy, QLabel)

from doocspie.pyqt.messages import show_warning


class PrintToolUi(QDialog):
    """Print tool ui class for providing print dialogs.

    This class provides the ui for providing dialogs for printing.
    """

    FATAL = "FATAL"
    _SEVERITY_LIST = ["NONE", FATAL, "ERROR", "FIXED", "TODO", "DONE", "WARN", "INFO", "IDEA", "DOCU", "MEASURE",
                      "STATISTICS", "WOW"]

    def __init__(self, window_title, all_printers, non_available_printers, default_author):
        """Constructor of the print tool ui class.

        This constructor initializes the print tool ui class using the window title and a default author.

        Args:
            window_title (str): The window title to show up in the dialog.
            all_printers (dict): All e-logs (key) and printers (value) requested for printing.
            non_available_printers (dict): The non-available e-logs (key) and printers (value).
            default_author (str): The default author to show up in the dialog and print.
        """
        super().__init__()
        self.setWindowTitle(window_title)
        self._main_layout = QVBoxLayout()
        self._author_line_edit = QLineEdit(default_author)
        self._title_line_edit = QLineEdit()
        self._severity_combo_box = self._get_combo_box_with(self._SEVERITY_LIST)
        self._printers_combo_box = self._get_combo_box_with(all_printers.keys())
        self._disable_non_available_printers(list(all_printers.keys()), non_available_printers)
        self._text_edit = QTextEdit()
        self._print_button = self._get_push_button_with("Print")
        self._cancel_button = self._get_push_button_with("Cancel")
        self._create_layout()

    @staticmethod
    def _get_combo_box_with(items):
        """Helper method to return a customized combo box."""
        combo_box = QComboBox()
        combo_box.addItems(items)
        return combo_box

    @staticmethod
    def _get_push_button_with(label):
        """Helper method to return a customized push button."""
        push_button = QPushButton(label)
        push_button.setAutoDefault(False)
        return push_button

    def _disable_non_available_printers(self, all_printers, non_available_printers):
        """Helper method to disable non-available printers"""
        to_disable = [all_printers.index(printer) for printer in non_available_printers]
        for index in to_disable:
            self._printers_combo_box.model().item(index).setEnabled(False)

        enabled = [printer for printer in all_printers if printer not in non_available_printers]
        if enabled:
            self._printers_combo_box.setCurrentIndex(all_printers.index(enabled[0]))
        else:
            show_warning("No e-log printer available on host")
            self._printers_combo_box.setEnabled(False)

    def _create_layout(self):
        """Helper method to invoke the multiple layout creators."""
        self._create_main_layout()
        self._create_info_layout()
        self._create_text_field_layout()
        self._create_buttons_layout()

    def _create_main_layout(self):
        """Helper method to create the print tool's main layout."""
        self.setLayout(self._main_layout)
        self.setModal(True)

    def _create_info_layout(self):
        """Helper method to create the print tool's info layout."""
        self._author_line_edit.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        info_layout = QHBoxLayout()
        info_layout.addLayout(self._get_vertical_layout_with("Author:", self._author_line_edit))
        info_layout.addLayout(self._get_vertical_layout_with("Title:", self._title_line_edit))
        info_layout.addLayout(self._get_vertical_layout_with("Severity:", self._severity_combo_box))
        info_layout.addLayout(self._get_vertical_layout_with("E-Log:", self._printers_combo_box))
        self._main_layout.addLayout(info_layout)

    @staticmethod
    def _get_vertical_layout_with(text, widget):
        """Helper method to create a vertical layout with a given text and widget."""
        layout = QVBoxLayout()
        layout.addWidget(QLabel(text))
        layout.addWidget(widget)
        return layout

    def _create_text_field_layout(self):
        """Helper method to create the print tool's text field layout."""
        self._text_edit.setFocus()
        self._main_layout.addWidget(self._text_edit)

    def _create_buttons_layout(self):
        """Helper method to create the print tool's button layout."""
        buttons_layout = QHBoxLayout()
        buttons_layout.addWidget(self._print_button)
        buttons_layout.addWidget(self._cancel_button)
        self._main_layout.addLayout(buttons_layout)

    @property
    def author_line_edit(self):
        """QLineEdit: The author field of the print dialog."""
        return self._author_line_edit

    @property
    def title_line_edit(self):
        """QLineEdit: The title field of the print dialog."""
        return self._title_line_edit

    @property
    def severity_combo_box(self):
        """QComboBox: The severity combo box of the print dialog."""
        return self._severity_combo_box

    @property
    def printers_combo_box(self):
        """QComboBox: The printers' combo box of the print dialog."""
        return self._printers_combo_box

    @property
    def text_edit(self):
        """QTextEdit: The text field of the print dialog."""
        return self._text_edit

    @property
    def print_button(self):
        """QPushButton: The print button of the print dialog."""
        return self._print_button

    @property
    def cancel_button(self):
        """QPushButton: The cancel button of the print dialog."""
        return self._cancel_button

    def show_window(self):
        """Showing the print tool dialog ui.

        Returns:
            None
        """
        self.show()
        self.setMinimumSize(self.width(), self.height())
