"""Module with print tool class for providing print dialogs.

This module provides the print tool class for providing dialogs for printing.
"""
from .print_tool_ui import PrintToolUi


class PrintTool:
    """Print tool class for providing print dialogs.

    This class provides the methods for providing dialogs for printing.
    """

    def __init__(self, print_service, default_author, version):
        """Constructor of the print tool class.

        This constructor initializes the print tool class using the e-log data, the default author,
        and a version number.

        Args:
            print_service (PrintService): The print service for printing to e-log.
            default_author (str): The default author to show up in the dialog and print.
            version (str): A version number to show up in the print.
        """
        self._print_tool_ui = PrintToolUi("E-Log Printer",
                                          all_printers=print_service.e_log_data.printers,
                                          non_available_printers=print_service.non_available_printers,
                                          default_author=default_author)
        self._e_log_data = print_service.e_log_data
        self._version = version
        self._create_connections()

    def _create_connections(self):
        """Helper method to create the required GUI connections."""
        self._print_tool_ui.print_button.clicked.connect(self._print_button_clicked)
        self._print_tool_ui.cancel_button.clicked.connect(self._cancel_button_clicked)

    def _print_button_clicked(self):
        """Helper method to realize the actions for clicking the 'print' button."""
        self._e_log_data.to_print = True
        self._e_log_data.author = self._print_tool_ui.author_line_edit.text()
        self._e_log_data.title = self._print_tool_ui.title_line_edit.text()
        self._e_log_data.severity = self._print_tool_ui.severity_combo_box.currentText()
        self._e_log_data.printer = self._e_log_data.printers[self._print_tool_ui.printers_combo_box.currentText()]
        self._e_log_data.text = self._print_tool_ui.text_edit.toPlainText() + "\n----Version " + self._version
        self._print_tool_ui.close()

    def _cancel_button_clicked(self):
        """Helper method to realize the actions for clicking the 'cancel' button."""
        self._print_tool_ui.close()

    def show_standard_dialog(self):
        """Showing a PyQt print dialog for the standard print dialog.

        Returns:
            None
        """
        self._e_log_data.to_print = False
        self._print_tool_ui.show_window()
        self._print_tool_ui.exec()

    def show_error_dialog_with(self, exception_name, message):
        """Showing a PyQt print dialog for the error print dialog.

         Args:
            exception_name (str): The exception name to show up in the error dialog's title.
            message (str): The message name to show up in the error dialog.

        Returns:
            None
        """
        self._print_tool_ui.title_line_edit.setText(exception_name)
        self._print_tool_ui.severity_combo_box.setCurrentText(PrintToolUi.FATAL)
        self._print_tool_ui.text_edit.setText(message)
        self.show_standard_dialog()
