"""Module with print service class for sending data to printers.

This module provides the print service class for sending data to printers.
"""
import subprocess
import sys

from doocspie.pyqt.messages import show_error


class PrintService:
    """Print service class for sending data to printers.

    This class provides the methods for updating and sending data to printers.
    """

    def __init__(self, e_log_data):
        """Constructor of the print service class.

        This constructor initializes the print service class using the e-log data.

        Args:
            e_log_data (ELogData): The e-log data to print.
        """
        self._e_log_data = e_log_data
        self._non_available_printers = self._get_non_available(self._e_log_data.printers)

    @staticmethod
    def _get_non_available(printers):
        """Helper method to get non-available e-log printers."""
        try:
            result = subprocess.run(["/usr/bin/lpstat", "-p"], check=False,
                                    stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        except FileNotFoundError:
            sys.exit(show_error("Cannot check available e-log printers"))
        all_printers = {p.split()[1] for p in result.stdout.decode("utf-8").split("\n")[:-1]}
        non_available_e_logs = set(printers.values()) - all_printers
        return [k for k, v in printers.items() if v in non_available_e_logs]

    @property
    def e_log_data(self):
        """ELogData: The e-log data instance."""
        return self._e_log_data

    @property
    def non_available_printers(self):
        """dict: The non-available e-log printers"""
        return self._non_available_printers

    def update_e_log_data_with(self, screenshot):
        """Updating the e-log data with a screenshot of the application.

         Args:
            screenshot (str): The application's screenshot encoded as a base64 string.

        Returns:
            None
        """
        self._e_log_data.image = screenshot

    def print(self):
        """Printing the e-log data to a particular printer.

        Returns:
            None
        """
        if self._e_log_data.to_print:
            try:
                with subprocess.Popen(["/usr/bin/lp", "-o", "raw", "-d", self._e_log_data.printer],
                                      stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE) as printer:
                    _, stderr = printer.communicate(self._e_log_data.get_e_log_entry())
                    if stderr:
                        raise FileNotFoundError
            except FileNotFoundError:
                show_error("Printing to e-log failed")
