"""Module with e-log printer class for providing e-log print dialogs.

This module offers the e-log printer class for providing dialogs for printing to e-logs.
"""
import sys
import traceback

from PyQt5.QtCore import QByteArray, QBuffer, QIODevice
from PyQt5.QtWidgets import QWidget

from doocspie.pyqt.e_log_printer.e_log_data import ELogData
from doocspie.pyqt.e_log_printer.print_service import PrintService
from doocspie.pyqt.e_log_printer.print_tool import PrintTool


class ELogPrinter:
    """E-log printer class for providing e-log print dialogs.

    This class offers the methods for providing dialogs for printing to e-logs.
    """

    def __init__(self, application_ui, printers, print_exceptions=True):
        """Constructor of the e-log printer class.

        This constructor initializes the instance with the application's ui, the printer (e-log) to print to and the
        optional 'print_exceptions' state.

        Args:
            application_ui (subclass of QMainWindow): The application's PyQt ui.
            printers (dict): The e-logs (key) and printers (value).
            print_exceptions (bool, optional): The optional state for handling exceptions with a print dialog.
        """
        self._application_ui = application_ui
        self._print_service = PrintService(ELogData(printers))
        self._print_tool = PrintTool(self._print_service, self._application_ui.application_name,
                                     self._application_ui.version)
        if print_exceptions:
            sys.excepthook = self._exception_handler

    def show_dialog(self):
        """Showing a PyQt print dialog of the particular application for printing to a particular e-log.

        Returns:
            None
        """
        self._show(self._print_tool.show_standard_dialog)

    def _show_error_dialog_with(self, exception_name, message):
        """Helper method to show the printer tool's dialog with its specific error dialog."""
        self._show(lambda: self._print_tool.show_error_dialog_with(exception_name, message))

    def _show(self, dialog):
        """Helper method to show the actual printer tool's dialog."""
        self._print_service.update_e_log_data_with(self._screenshot())
        dialog()
        self._print_service.print()

    def _screenshot(self):
        """Helper method to grab and return a screenshot of the actual widget."""
        screenshot_tmp = QByteArray()
        screenshot_buffer = QBuffer(screenshot_tmp)
        screenshot_buffer.open(QIODevice.WriteOnly)
        widget = QWidget.grab(self._application_ui)
        widget.save(screenshot_buffer, "png")
        return screenshot_tmp.toBase64().data().decode()

    def _exception_handler(self, exception_type, exception_value, exception_traceback):
        """Helper method to handle unexpected exceptions with an error dialog."""
        self._application_ui.close()  # gives extra visual hint for erroneous situation
        exception = exception_type.__name__ + ": unexpected exception occurred."
        message = "Traceback (most recent call last):\n"
        message += "".join(traceback.format_list(traceback.extract_tb(exception_traceback))) + "\n"
        message += exception_type.__name__ + ": " + str(exception_value)
        sys.exit(self._show_error_dialog_with(exception, message))
