"""Module with property server class for simplified access to the property servers.

This module provides the property server class for simplified access to property servers at FLASH.
"""
import doocspie


class PropertyServer:
    """PropertyServer class for simplified access to the property servers.

    This class provides the property server class for simplified access to property servers at FLASH.
    """
    _FACILITY_DEVICE_PREFIX = "FLASH.EXP/USER.STORE."

    def get_locations(self, full_address=False):
        """Get all available property server locations with the option to return the full addresses.

        Args:
            full_address (bool, optional): The optional state for returning the full addresses.

        Returns:
            tuple: All available property server locations.
        """
        locations = tuple(device.split(".")[-1] for device in doocspie.ls(self._FACILITY_DEVICE_PREFIX + "*"))
        if full_address:
            return tuple(self._FACILITY_DEVICE_PREFIX + location + "/" + location for location in locations)
        return locations

    def get_properties_of_location(self, location, full_address=False):
        """Get all available property server properties with the option to return the full addresses.

        Args:
            location (str): The property server location to request the properties from.
            full_address (bool, optional): The optional state for returning the full addresses.

        Returns:
            tuple: All available property server properties.

        Raises:
            ValueError: Exception for non-existing location.
        """
        if location.startswith(self._FACILITY_DEVICE_PREFIX):
            full_location = location
        else:
            full_location = self._FACILITY_DEVICE_PREFIX + location + "/" + location + "/"

        try:
            properties = tuple(property.split("/")[-1] for property in doocspie.ls(full_location))
        except doocspie.DoocspieException:
            raise ValueError("location '" + location + "' does not exist") from None

        filtered_properties = filter(self._get_relevant, properties)
        if full_address:
            return tuple(full_location + property for property in filtered_properties)
        return tuple(filtered_properties)

    @staticmethod
    def _get_relevant(property):
        """Helper method to get only the relevant properties, i.e., those starting with 'VAL' or 'ARRAY'."""
        return property.startswith("VAL") or property.startswith("ARRAY")

    def get(self, location, property):
        """Get the readout from the property server for the given location and property.

        Args:
            location (str): The property server location to request the readout for the property from.
            property (str): The actual property to request the readout from.

        Returns:
            Readout: An instance of the readout object from the property server for the given location and property.

        Raises:
            ValueError: Exception for non-existing location and/or property or not supported full address input.
            DoocspieException: Doocspie related exception for unexpected events.
        """
        self._check_if_full_address(location, property)
        try:
            return doocspie.get(self._FACILITY_DEVICE_PREFIX + location + "/" + location + "/" + property)
        except doocspie.DoocspieException as exc:
            raise self._check_valid_location_and_property(location, property, exc) from None

    def set(self, location, property, value, allow_resizing=False):
        """Set the property server content for the given location and property with the given value.

        Args:
            location (str): The property server location of the property to set the value to.
            property (str): The actual property to set the value to.
            value (property dependent): The value to set the property to.
            allow_resizing(bool, optional): The optional state for allowing resizing of array-like DOOCS types.

        Returns:
            None

        Raises:
            ValueError: Exception non-existing location and/or property or for not supported full address input.
            DoocspieException: Doocspie related exception for unexpected events.
        """
        self._check_if_full_address(location, property)
        try:
            doocspie.set(self._FACILITY_DEVICE_PREFIX + location + "/" + location + "/" + property, value,
                         allow_resizing)
        except doocspie.DoocspieException as exc:
            raise self._check_valid_location_and_property(location, property, exc) from None

    def _check_if_full_address(self, location, property):
        """Helper method to check if location and/or property start with the full address."""
        if location.startswith(self._FACILITY_DEVICE_PREFIX) or property.startswith(self._FACILITY_DEVICE_PREFIX):
            raise ValueError("full address of location and/or property is not supported")

    def _check_valid_location_and_property(self, location, property, exc):
        """Helper method to check if location and/or property are valid."""
        if location not in self.get_locations():
            return ValueError("location '" + location + "' does not exist")
        if property not in self.get_properties_of_location(location):
            return ValueError("property '" + property + "' does not exist")
        return exc
