"""Module with viewer tool ui class for viewing data.

This module provides the viewer tool ui class for providing a tool for viewing data.
"""
from PyQt5.QtCore import Qt, QRegExp
from PyQt5.QtGui import QRegExpValidator
from PyQt5.QtWidgets import (QMainWindow, QVBoxLayout, QWidget, QPushButton, QSizePolicy, QHBoxLayout, QGroupBox,
                             QCheckBox, QLabel, QComboBox, QStackedWidget, QProgressBar, QLineEdit)

from ..plotter.viewer_plotter import ViewerPlotter


class ViewerToolUi(QMainWindow):
    """Viewer tool ui class for viewing data.

    This class provides the ui for providing a tool for viewing data.
    """

    IMAGE = ViewerPlotter.IMAGE
    VECTOR = ViewerPlotter.VECTOR
    DUAL_VECTOR = ViewerPlotter.DUAL_VECTOR

    START_LABEL = "Start"

    def __init__(self, *sources):
        """Constructor of the viewer tool ui class.

        This constructor initializes the viewer tool ui class.

        Args:
            sources (tuple): The data sources to visualize.
        """
        super().__init__()
        self._name = "Viewer"
        self._viewer_plotter = ViewerPlotter()
        self._main_layout = QVBoxLayout()
        self._start_button = self._get_labeled_push_button(self.START_LABEL)
        self._stop_button = self._get_labeled_push_button("Stop")
        self._averaging_check_box = QCheckBox("Averaging")
        self._average_samples = QLabel("0")
        self._take_button_and_progress_bar_stack = QStackedWidget()
        self._take_button = self._get_labeled_push_button("Take")
        self._progress_bar = QProgressBar()
        self._background_samples_line_edit = self._get_sized_line_edit()
        self._apply_background_correction_check_box = QCheckBox("Apply", enabled=False)
        self._source_combo_box = self._get_combo_box_with(sources)
        self._create_layout()
        self._create_connections()

    @staticmethod
    def _get_labeled_push_button(label):
        """Helper method to return a labeled push button."""
        push_button = QPushButton(label)
        push_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        return push_button

    @staticmethod
    def _get_sized_line_edit():
        """Helper method to return a sized line edit."""
        samples_line_edit = QLineEdit("10")
        samples_line_edit.setAlignment(Qt.AlignRight)
        samples_line_edit.setMaxLength(3)
        samples_line_edit.setMaximumWidth(33)
        samples_line_edit.setValidator(QRegExpValidator(QRegExp("[1-9][0-9]*")))
        return samples_line_edit

    @staticmethod
    def _get_combo_box_with(sources_of_sources):
        """Helper method to return a customized combo box."""
        combo_box = QComboBox()
        for sources in sources_of_sources:
            combo_box.addItems(sources)
        return combo_box

    def _create_layout(self):
        """Helper method to invoke the multiple layout creators."""
        self._create_main_layout()
        self._create_plot_layout()
        self._create_controls_layout()

    def _create_main_layout(self):
        """Helper method to create the viewer tool's main layout."""
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_plot_layout(self):
        """Helper method to create the viewer tool's plot layout."""
        plot_layout = QHBoxLayout()
        plot_layout.addWidget(self._viewer_plotter.widget)
        self._main_layout.addLayout(plot_layout)

    def _create_controls_layout(self):
        """Helper method to create the viewer tool's control layout."""
        controls_layout = QHBoxLayout()
        controls_layout.addWidget(self._create_control_group_box())
        controls_layout.addWidget(self._create_background_correction_group_box())
        controls_layout.addWidget(self._create_source_group_box())
        self._main_layout.addLayout(controls_layout)

    def _create_control_group_box(self):
        """Helper method to create the control group's layout."""
        group_box = QGroupBox("Control")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._start_button)
        widgets_layout.addWidget(self._stop_button)
        widgets_layout.addWidget(self._averaging_check_box)
        widgets_layout.addWidget(QLabel("Samples:"))
        widgets_layout.addWidget(self._average_samples)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_background_correction_group_box(self):
        """Helper method to create the background correction group's layout."""
        group_box = QGroupBox("Background Correction")
        group_box.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        widgets_layout = QHBoxLayout()
        self._take_button_and_progress_bar_stack.addWidget(self._take_button)
        self._take_button_and_progress_bar_stack.addWidget(self._progress_bar)
        widgets_layout.addWidget(self._take_button_and_progress_bar_stack)
        widgets_layout.addWidget(QLabel("Samples:"))
        widgets_layout.addWidget(self._background_samples_line_edit)
        widgets_layout.addWidget(self._apply_background_correction_check_box)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_source_group_box(self):
        """Helper method to create the source group's layout."""
        group_box = QGroupBox("Source")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._source_combo_box)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_connections(self):
        """Helper method to create the required GUI connections."""
        self._apply_background_correction_check_box.stateChanged.connect(self._viewer_plotter.reset_plot_range)

    @property
    def name(self):
        """str: The viewer tool's name."""
        return self._name

    @property
    def start_button(self):
        """QPushButton: The start button of the viewer tool."""
        return self._start_button

    @property
    def stop_button(self):
        """QPushButton: The stop button of the viewer tool."""
        return self._stop_button

    @property
    def averaging_check_box(self):
        """QCheckBox: The averaging checkbox of the viewer tool."""
        return self._averaging_check_box

    @property
    def average_samples(self):
        """QLabel: The average samples label of the viewer tool."""
        return self._average_samples

    @property
    def take_button(self):
        """QPushButton: The take button of the viewer tool."""
        return self._take_button

    @property
    def background_samples_line_edit(self):
        """QLineEdit: The background samples line edit of the viewer tool."""
        return self._background_samples_line_edit

    @property
    def apply_background_correction_check_box(self):
        """QCheckBox: The apply background correction checkbox of the viewer tool."""
        return self._apply_background_correction_check_box

    @property
    def source_combo_box(self):
        """QComboBox: The source combo box of the viewer tool."""
        return self._source_combo_box

    def set_plot_type(self, plot_type):
        """Setting the plot type of actual viewer plotter.

        Args:
            plot_type (str): The plot type (either IMAGE or VECTOR) to set.

        Returns:
            None
        """
        self._viewer_plotter.set_plot_type(plot_type)

    def show_progress_bar(self):
        """Showing the progress bar.

        Returns:
            None
        """
        self._progress_bar.setValue(0)
        self._take_button_and_progress_bar_stack.setCurrentWidget(self._progress_bar)

    def show_take_button(self):
        """Showing the take button.

        Returns:
            None
        """
        self._take_button_and_progress_bar_stack.setCurrentWidget(self._take_button)

    def plot(self, data, axis_labels):
        """Plotting the given data with axis labels.

        Args:
            data (property dependent): The property dependent data to plot.
            axis_labels (tuple): The axis labels for the data.

        Returns:
            None
        """
        self._viewer_plotter.set(data, axis_labels)

    def reset_plot_range(self):
        """Resetting the plot range.

        Returns:
            None
        """
        self._viewer_plotter.reset_plot_range()

    def set_progress(self, value):
        """Setting the progress of the progress bar.

        Returns:
            None
        """
        self._progress_bar.setValue(value)
