"""Module with viewer tool class for viewing data.

This module provides the viewer tool class for providing a tool for viewing data.
"""
import numpy as np
from doocspie.pyqt import messages

from .viewer_tool_ui import ViewerToolUi


class ViewerTool:
    """Viewer tool class for viewing data.

    This class provides the methods for providing a tool for viewing data.
    """

    def __init__(self, image_data, vector_data, dual_vector_data, axis_labels):
        """Constructor of the viewer tool class.

        This constructor initializes the viewer tool class.

        Args:
            image_data (ImageData): The image data of the readout sources.
            vector_data (VectorData): The vector data of the readout sources.
            dual_vector_data (DualVectorData): The dual vector data of the readout sources.
            axis_labels (dict): The axis labels for the data sources.
        """
        self._viewer_tool_ui = ViewerToolUi(image_data.get_sources(), vector_data.get_sources(),
                                            dual_vector_data.get_sources())
        self._image_data = image_data
        self._vector_data = vector_data
        self._dual_vector_data = dual_vector_data
        self._data = {self._image_data: self._viewer_tool_ui.IMAGE,
                      self._vector_data: self._viewer_tool_ui.VECTOR,
                      self._dual_vector_data: self._viewer_tool_ui.DUAL_VECTOR}
        self._axis_labels = axis_labels
        self._selected_source = None
        self._selected_data = None
        self._is_started = False
        self._create_connections()
        self._select_source_and_data_to_plot()

    def _create_connections(self):
        """Helper method to create the required GUI connections."""
        self._viewer_tool_ui.start_button.clicked.connect(self._start_button_clicked)
        self._viewer_tool_ui.take_button.clicked.connect(self._take_button_clicked)
        self._viewer_tool_ui.stop_button.clicked.connect(self._stop_button_clicked)
        self._viewer_tool_ui.source_combo_box.currentIndexChanged.connect(self._viewer_tool_ui.reset_plot_range)
        self._viewer_tool_ui.source_combo_box.currentIndexChanged.connect(self._select_source_and_data_to_plot)
        self._viewer_tool_ui.source_combo_box.currentIndexChanged.connect(self._offline_update)
        self._viewer_tool_ui.source_combo_box.currentIndexChanged.connect(self._viewer_tool_ui.reset_plot_range)
        self._viewer_tool_ui.averaging_check_box.stateChanged.connect(self._set_averaging)
        self._viewer_tool_ui.apply_background_correction_check_box.stateChanged.connect(self._set_background_correction)

    @property
    def ui(self):
        """ViewerToolUi: The viewer tool's user interface."""
        return self._viewer_tool_ui

    @property
    def is_started(self):
        """bool: The running state of the GUI."""
        return self._is_started

    def _set_averaging(self, code):
        """Helper method to set averaging according to the given code."""
        state = bool(code)
        for data in self._data:
            data.set_averaging(state)

    def _set_background_correction(self, code):
        """Helper method to set background correction according to the given code."""
        state = bool(code)
        for data in self._data:
            data.set_background_correction(state)

    def _start_button_clicked(self):
        """Helper method to realize the actions for clicking the 'start' button."""
        self._is_started = True
        self._viewer_tool_ui.start_button.setText("running ...")
        self._viewer_tool_ui.start_button.setEnabled(False)
        self._viewer_tool_ui.take_button.setEnabled(False)

    def _take_button_clicked(self):
        """Helper method to realize the actions for clicking the 'take' button."""
        if self._is_input_validation_successful():
            self._is_started = True
            self._viewer_tool_ui.start_button.setEnabled(False)
            self._viewer_tool_ui.show_progress_bar()
            self._viewer_tool_ui.background_samples_line_edit.setEnabled(False)
            self._viewer_tool_ui.averaging_check_box.setChecked(False)
            self.set_apply_background_correction_check_box_to(False)
            self._set_background_taking(int(self._viewer_tool_ui.background_samples_line_edit.text()))

    def _is_input_validation_successful(self):
        """Helper method to validate the input."""
        try:
            int(self._viewer_tool_ui.background_samples_line_edit.text())
        except ValueError:
            self.set_apply_background_correction_check_box_to(False)
            messages.show_error("Empty 'Samples' field")
            return False
        return True

    def _set_background_taking(self, samples):
        """Helper method to set background-taking with given samples."""
        for data in self._data:
            data.set_background_taking(samples)

    def _stop_button_clicked(self):
        """Helper method to realize the actions for clicking the 'stop' button."""
        if self._is_started:
            self._reset_background_taking()
            self._viewer_tool_ui.start_button.setText(self._viewer_tool_ui.START_LABEL)
            self._viewer_tool_ui.start_button.setEnabled(True)
            self._viewer_tool_ui.take_button.setEnabled(True)
            self._viewer_tool_ui.background_samples_line_edit.setEnabled(True)
            self._viewer_tool_ui.show_take_button()
            self._is_started = False

    def _reset_background_taking(self):
        """Helper method to reset background-taking."""
        for data in self._data:
            data.reset_background_taking()

    def _select_source_and_data_to_plot(self):
        """Helper method to select the source and data to plot."""
        self._selected_source = self._viewer_tool_ui.source_combo_box.currentText()
        for data, plot_type in self._data.items():
            if self._selected_source in data.get_sources():
                self._selected_data = data
                self._viewer_tool_ui.set_plot_type(plot_type)
                return
        self._viewer_tool_ui.start_button.setEnabled(False)
        self._viewer_tool_ui.take_button.setEnabled(False)

    def set_apply_background_correction_check_box_to(self, state):
        """Setting the 'apply background correction' check box to the given state.

        Args:
            state (bool): The state of applying background correction.

        Returns:
            None
        """
        self._viewer_tool_ui.apply_background_correction_check_box.setChecked(state)
        self._viewer_tool_ui.apply_background_correction_check_box.setEnabled(state)

    def update(self, to_plot):
        """Updating the viewer tool.

        Args:
            to_plot (bool): The state of plotting.

        Returns:
            None
        """
        if self._is_started:
            if to_plot:
                self._plot()
                self._viewer_tool_ui.average_samples.setText(
                    self._selected_data.get_average_samples(self._selected_source))

            if self._selected_data.is_taking_background:
                self._update_background_taking_progress()

    def _update_background_taking_progress(self):
        """Helper method to update the background-taking progress."""
        self._viewer_tool_ui.set_progress(self._selected_data.get_background_taking_progress(self._selected_source))
        if self._selected_data.get_background_taking_progress(self._selected_source) == 100:
            self.set_apply_background_correction_check_box_to(True)
            self._viewer_tool_ui.stop_button.click()

    def _offline_update(self):
        """Helper method to offline-update the viewer tool."""
        if not (self._is_started or self._selected_data.get(self._selected_source) is None):
            self._plot()

    def _plot(self):
        """Helper method to plot the selected data with given labels."""
        self._viewer_tool_ui.plot(self._selected_data.get(self._selected_source),
                                  self._axis_labels.get(self._selected_source, (None, None)))

    def get_export_content(self):
        """Getting the content to export to a file.

        Returns:
            dict
        """
        content = {"selected source": [self._selected_source],
                   "axis label x": [self._axis_labels.get(self._selected_source, (None, None))[0]],
                   "axis label y": [self._axis_labels.get(self._selected_source, (None, None))[1]],
                   "averaging": [bool(int(self._viewer_tool_ui.average_samples.text()))],
                   "average samples": [int(self._viewer_tool_ui.average_samples.text())]}

        if self._selected_source:
            selected_data = self._selected_data.get(self._selected_source)
            if selected_data is not None:
                source_data_x = "source data x"
                source_data_y = "source data y"
                if self._selected_source in self._image_data.get_sources():
                    content["source data image"] = selected_data.flatten()
                    content["image shape 0"] = [selected_data.shape[0]]
                    content["image shape 1"] = [selected_data.shape[1]]
                elif self._selected_source in self._vector_data.get_sources():
                    content[source_data_x] = np.arange(len(selected_data))
                    content[source_data_y] = selected_data
                elif self._selected_source in self._dual_vector_data.get_sources():
                    content[source_data_x] = selected_data[0]
                    content[source_data_y] = selected_data[1]
                content["background corrected"] = [self._selected_data.is_background_corrected]
                content["background samples"] = [int(self._viewer_tool_ui.background_samples_line_edit.text())]
        return content
