"""Module with logger tool ui class for logging data.

This module provides the logger tool ui class for providing a tool for logging data.
"""
from PyQt5.QtCore import Qt, QRegExp
from PyQt5.QtGui import QRegExpValidator
from PyQt5.QtWidgets import (QMainWindow, QVBoxLayout, QWidget, QPushButton, QSizePolicy, QHBoxLayout, QGroupBox,
                             QLineEdit, QCheckBox, QComboBox, QLabel)

from ..plotter.logger_plotter import LoggerPlotter


class LoggerToolUi(QMainWindow):
    """Logger tool ui class for logging data.

    This class provides the ui for providing a tool for logging data.
    """

    START_LABEL = "Start"

    def __init__(self, sources):
        """Constructor of the logger tool ui class.

        This constructor initializes the logger tool ui class.

        Args:
            sources (tuple): The data sources to visualize.
        """
        super().__init__()
        self._name = "Logger"
        self._logger_plotter = LoggerPlotter(sources)
        self._main_layout = QVBoxLayout()
        self._start_button = self._get_labeled_push_button(self.START_LABEL)
        self._start_button.setEnabled(bool(sources))
        self._stop_button = self._get_labeled_push_button("Stop")
        self._reset_button = self._get_labeled_push_button("Reset")
        self._use_timestamps_checkbox = QCheckBox("Use Timestamps")
        self._enable_averaging_checkbox = QCheckBox("Enable")
        self._samples_line_edit = self._get_samples_line_edit()
        self._enable_movable_line_checkbox = QCheckBox("Enable")
        self._source_combo_box = self._get_combo_box_with(sources)
        self._create_layout()
        self._create_connections()

    @staticmethod
    def _get_labeled_push_button(label):
        """Helper method to return a labeled push button."""
        push_button = QPushButton(label)
        push_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        return push_button

    @staticmethod
    def _get_samples_line_edit():
        """Helper method to return the 'samples' line edit."""
        line_edit = QLineEdit("10")
        line_edit.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        line_edit.setAlignment(Qt.AlignRight)
        line_edit.setValidator(QRegExpValidator(QRegExp("[1-9][0-9]*")))
        line_edit.setMaxLength(3)
        line_edit.setMaximumWidth(33)
        return line_edit

    @staticmethod
    def _get_combo_box_with(sources):
        """Helper method to return a customized combo box."""
        combo_box = QComboBox()
        combo_box.addItems(sources)
        return combo_box

    def _create_layout(self):
        """Helper method to invoke the multiple layout creators."""
        self._create_main_layout()
        self._create_plot_layout()
        self._create_controls_layout()

    def _create_main_layout(self):
        """Helper method to create the logger tool's main layout."""
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_plot_layout(self):
        """Helper method to create the logger tool's plot layout."""
        plot_layout = QHBoxLayout()
        plot_layout.addWidget(self._logger_plotter.widget)
        self._main_layout.addLayout(plot_layout)

    def _create_controls_layout(self):
        """Helper method to create the logger tool's control layout."""
        controls_layout = QHBoxLayout()
        controls_layout.addWidget(self._create_control_group_box())
        controls_layout.addWidget(self._create_averaging_group_box())
        controls_layout.addWidget(self._create_line_marker_group_box())
        controls_layout.addWidget(self._create_source_group_box())
        self._main_layout.addLayout(controls_layout)

    def _create_control_group_box(self):
        """Helper method to create the control group's layout."""
        group_box = QGroupBox("Control")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._start_button)
        widgets_layout.addWidget(self._stop_button)
        widgets_layout.addWidget(self._reset_button)
        widgets_layout.addWidget(self._use_timestamps_checkbox)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_averaging_group_box(self):
        """Helper method to create the averaging group's layout."""
        group_box = QGroupBox("Moving Average")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._enable_averaging_checkbox)
        widgets_layout.addWidget(QLabel("Samples:"))
        widgets_layout.addWidget(self._samples_line_edit)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_line_marker_group_box(self):
        """Helper method to create the line marker group's layout."""
        group_box = QGroupBox("Line Marker")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._enable_movable_line_checkbox)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_source_group_box(self):
        """Helper method to create the source group's layout."""
        group_box = QGroupBox("Source")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._source_combo_box)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_connections(self):
        """Helper method to create connections."""
        self._reset_button.clicked.connect(self._logger_plotter.reset)
        self._enable_movable_line_checkbox.stateChanged.connect(self._logger_plotter.set_movable_line_visible)
        self._source_combo_box.currentIndexChanged.connect(
            lambda: self._logger_plotter.set_movable_line(self._source_combo_box.currentText()))

    @property
    def name(self):
        """str: The logger tool's name."""
        return self._name

    @property
    def start_button(self):
        """QPushButton: The start button of the logger tool."""
        return self._start_button

    @property
    def stop_button(self):
        """QPushButton: The stop button of the logger tool."""
        return self._stop_button

    @property
    def reset_button(self):
        """QPushButton: The reset button of the logger tool."""
        return self._reset_button

    @property
    def use_timestamps_checkbox(self):
        """QCheckBox: The use timestamps checkbox of the logger tool."""
        return self._use_timestamps_checkbox

    @property
    def enable_averaging_checkbox(self):
        """QCheckBox: The enable averaging checkbox of the logger tool."""
        return self._enable_averaging_checkbox

    @property
    def samples_line_edit(self):
        """QLineEdit: The samples line edit of the logger tool."""
        return self._samples_line_edit

    @property
    def source_combo_box(self):
        """QComboBox: The source combo box of the logger tool."""
        return self._source_combo_box

    @property
    def movable_line_position_value(self):
        """float: The position value of the movable line."""
        return self._logger_plotter.movable_line_position_value

    def plot(self, data, axis_label, selected_source):
        """Plotting the given data with axis labels, and showing a movable line.

        Args:
            data (tuple): The data to plot.
            axis_label (str): The axis label for the data.
            selected_source (str): The selected data source for the movable line.

        Returns:
            None
        """
        self._logger_plotter.set(data, axis_label, selected_source)

    def reset_plot_range(self):
        """Resetting the plot range.

        Returns:
            None
        """
        self._logger_plotter.reset_plot_range()

    def use_timestamps(self, state):
        """Using the timestamps according to the state.

        Args:
            state (bool): The usage state of timestamps.

        Returns:
            None
        """
        self._logger_plotter.use_timestamps(state)

    def reset_movable_line(self):
        """Resetting the movable line.

        Returns:
            None
        """
        self._logger_plotter.reset()
