"""Module with logger tool class for logging data.

This module provides the logger tool class for providing a tool for logging data.
"""
import time
from collections import defaultdict

import numpy as np
from scipy.ndimage import uniform_filter1d
from doocspie.pyqt import messages

from .logger_tool_ui import LoggerToolUi


class LoggerTool:
    """Logger tool class for logging data.

    This class provides the methods for providing a tool for logging data.
    """

    def __init__(self, scalar_data, axis_labels):
        """Constructor of the logger tool class.

        This constructor initializes the logger tool class.

        Args:
            scalar_data (ScalarData): The scalar data of the readout sources.
            axis_labels (dict): The axis labels for the data sources.
        """
        self._logger_tool_ui = LoggerToolUi(scalar_data.get_sources())
        self._scalar_data = scalar_data
        self._axis_labels = axis_labels
        self._actual_data = defaultdict(list)
        self._selected_source = None
        self._is_started = False
        self._samples = None
        self._time_stamps = []
        self._create_connections()
        self._select_source_to_plot()
        self._plotted_once = False

    def _create_connections(self):
        """Helper method to create the required GUI connections."""
        self._logger_tool_ui.start_button.clicked.connect(self._start_button_clicked)
        self._logger_tool_ui.stop_button.clicked.connect(self._stop_button_clicked)
        self._logger_tool_ui.reset_button.clicked.connect(self._reset_button_clicked)
        self._logger_tool_ui.source_combo_box.currentIndexChanged.connect(self._select_source_to_plot)
        self._logger_tool_ui.source_combo_box.currentIndexChanged.connect(self._offline_update)
        self._logger_tool_ui.source_combo_box.currentIndexChanged.connect(self._logger_tool_ui.reset_plot_range)
        self._logger_tool_ui.enable_averaging_checkbox.clicked.connect(self._validate_input)
        self._logger_tool_ui.enable_averaging_checkbox.clicked.connect(self._offline_update)
        self._logger_tool_ui.enable_averaging_checkbox.clicked.connect(self._logger_tool_ui.reset_movable_line)
        self._logger_tool_ui.samples_line_edit.textChanged.connect(self._validate_input_if_enabled)
        self._logger_tool_ui.samples_line_edit.textChanged.connect(self._offline_update)
        self._logger_tool_ui.use_timestamps_checkbox.toggled.connect(self._logger_tool_ui.use_timestamps)
        self._logger_tool_ui.use_timestamps_checkbox.toggled.connect(self._offline_update)

    @property
    def ui(self):
        """LoggerToolUi: The logger tool user interface."""
        return self._logger_tool_ui

    @property
    def is_started(self):
        """bool: The running state of the GUI."""
        return self._is_started

    def _validate_input(self):
        """Helper method to validate averaging samples input."""
        try:
            self._samples = int(self._logger_tool_ui.samples_line_edit.text())
        except ValueError:
            self._logger_tool_ui.enable_averaging_checkbox.setChecked(False)
            messages.show_error("Empty 'Samples' field")

    def _validate_input_if_enabled(self):
        """Helper method to validate averaging samples input if enabled."""
        if self._logger_tool_ui.enable_averaging_checkbox.isChecked():
            self._validate_input()

    def _start_button_clicked(self):
        """Helper method to realize the actions for clicking the 'start' button."""
        self._is_started = True
        self._logger_tool_ui.start_button.setText("running ...")
        self._logger_tool_ui.start_button.setEnabled(False)

    def _stop_button_clicked(self):
        """Helper method to realize the actions for clicking the 'stop' button."""
        if self._is_started:
            self._logger_tool_ui.start_button.setText(self._logger_tool_ui.START_LABEL)
            self._logger_tool_ui.start_button.setEnabled(True)
            self._is_started = False

    def _reset_button_clicked(self):
        """Helper method to realize the actions for clicking the 'reset' button."""
        if not self._is_started and self._actual_data[self._selected_source]:
            self._logger_tool_ui.plot(([], []), "", self._selected_source)
        self._actual_data = defaultdict(list)
        self._time_stamps.clear()
        self._plotted_once = False

    def _select_source_to_plot(self):
        """Helper method to select the source to plot."""
        self._selected_source = self._logger_tool_ui.source_combo_box.currentText()

    def update(self, to_plot):
        """Updating the logger tool.

        Returns:
            None
        """
        if self._is_started:
            self._append_actual_data()
            self._time_stamps.append(time.time())
            if to_plot:
                self._plot()
                if not self._plotted_once:
                    self._plotted_once = True
                    self._logger_tool_ui.use_timestamps(self._logger_tool_ui.use_timestamps_checkbox.isChecked())

    def _append_actual_data(self):
        """Helper method to append actual data."""
        for source in self._scalar_data.get_sources():
            self._actual_data[source].append(self._scalar_data.get(source))

    def _offline_update(self):
        """Helper method to offline update the logger tool."""
        if not self._is_started and self._actual_data[self._selected_source]:
            self._plot()

    def _plot(self):
        """Helper method to plot the selected data with given labels."""
        self._logger_tool_ui.plot(self._get_data(), self._axis_labels.get(self._selected_source, (None, None))[1],
                                  self._selected_source)

    def _get_data(self):
        """Helper method to get the data."""
        data = self._actual_data[self._selected_source]
        if self._logger_tool_ui.enable_averaging_checkbox.isChecked():
            data = self._average(data)
        if self._logger_tool_ui.use_timestamps_checkbox.isChecked():
            return self._time_stamps, data
        return np.arange(1, len(data) + 1), data

    def _average(self, data):
        """Helper method to average the given data."""
        return uniform_filter1d(data, size=self._samples)

    def get_export_content(self):
        """Getting the content to export to a file.

        Returns:
            dict
        """
        return {"selected source": [self._selected_source],
                "axis label": [self._axis_labels.get(self._selected_source, (None, None))[1]],
                "timestamps used": [self._logger_tool_ui.use_timestamps_checkbox.isChecked()],
                "x axis": self._get_data()[0],
                "source data": self._get_data()[1],
                "movable line position": [self._logger_tool_ui.movable_line_position_value],
                "averaging enabled": [self._logger_tool_ui.enable_averaging_checkbox.isChecked()],
                "averaging samples": [self._logger_tool_ui.samples_line_edit.text()]}
