"""Module with datalyzer tool ui class for visualizing data.

This module provides the datalyzer tool ui class for providing a tool for visualizing data.
"""
from PyQt5.QtWidgets import QMainWindow, QVBoxLayout, QWidget, QAction, QTabWidget

from doocspie.pyqt import messages


class DatalyzerToolUi(QMainWindow):
    """Datalyzer tool ui class for visualizing data.

    This class provides the ui for providing a tool for visualizing data.
    """

    def __init__(self, application_name, version, *tool_uis):
        """Constructor of the datalyzer ui class.

        This constructor initializes the datalyzer ui class using the application's name and version,
        and its tool uis.

        Args:
            application_name (str): The application's name.
            version (str): The application's version.
            tool_uis (tuple): The uis of the different tools of the datalyzer.
        """
        super().__init__()
        self._application_name = application_name
        self._version = version
        self._tool_uis = tool_uis
        self._main_layout = QVBoxLayout()
        self._print_to_e_log_action = QAction("Print to E-Log")
        self._export_action = QAction("Export")
        self._click_about_action = QAction("About")
        self._tab_widget = QTabWidget()
        self._create_layout()
        self._create_connections()

    def _create_layout(self):
        """Helper method to invoke the multiple layout creators."""
        self._create_main_layout()
        self._create_menu_bar()
        self._create_tools_tab_widget()

    def _create_main_layout(self):
        """Helper method to create the datalyzer tool's main layout."""
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_menu_bar(self):
        """Helper method to create the menu bar."""
        self._create_file_menu()
        self._create_help_menu()

    def _create_file_menu(self):
        """Helper method to create the file menu."""
        file_menu = self.menuBar().addMenu("File")
        file_menu.addAction(self._print_to_e_log_action)
        file_menu.addAction(self._export_action)

    def _create_help_menu(self):
        """Helper method to create the help menu."""
        self.menuBar().addMenu("Help").addAction(self._click_about_action)

    def _create_tools_tab_widget(self):
        """Helper method to create the tab widgets."""
        for tool_ui in self._tool_uis:
            self._tab_widget.addTab(tool_ui, tool_ui.name)
        self._main_layout.addWidget(self._tab_widget)

    def _create_connections(self):
        """Helper method to create the connections."""
        self._click_about_action.triggered.connect(
            lambda: messages.show_about(self, self._application_name, self._version))

    @property
    def application_name(self):
        """str: The application's name."""
        return self._application_name

    @property
    def version(self):
        """str: The application's version."""
        return self._version

    @property
    def print_to_e_log_action(self):
        """QAction: The application's e-log print action."""
        return self._print_to_e_log_action

    @property
    def export_action(self):
        """QAction: The application's data export action."""
        return self._export_action

    def get_current_ui(self):
        """Returning the current tool ui.

        Returns:
            Derived class from QMainWindow
        """
        return self._tab_widget.currentWidget()

    def show_window(self):
        """Showing the datalyzer tool ui.

        Returns:
            None
        """
        self.show()
        self.setMinimumSize(self.width(), self.height())
