"""Module with datalyzer tool class for visualizing data.

This module provides the datalyzer tool class for providing a tool for visualizing data.
"""
from doocspie.pyqt import ELogPrinter, Timer, messages, ExportService

from .viewer_tool import ViewerTool
from .logger_tool import LoggerTool
from .correlator_tool import CorrelatorTool
from .datalyzer_tool_ui import DatalyzerToolUi


class DatalyzerTool:
    """Datalyzer tool class for visualizing data.

    This class provides the methods for providing a tool for visualizing data.
    """

    def __init__(self, application_name, export_directory, version, elogs, io_service, axis_labels):
        """Constructor of the datalyzer class.

        This constructor initializes the datalyzer class using the application's name and version,
        the e-logs to print to, and the io and export services, respectively.

        Args:
            application_name (str): The application's name.
            export_directory (str): The application's file export directory.
            version (str): The application's version.
            elogs (dict): The e-logs (key) and printers (value) available for printing.
            io_service (IOService): The io service for data input/output.
            axis_labels (dict): The axis labels for the data sources.
        """
        self._viewer_tool = ViewerTool(io_service.image_data, io_service.vector_data, io_service.dual_vector_data,
                                       axis_labels)
        self._logger_tool = LoggerTool(io_service.scalar_data, axis_labels)
        self._correlator_tool = CorrelatorTool(io_service.scalar_data, axis_labels)
        self._datalyzer_tool_ui = DatalyzerToolUi(application_name, version,
                                                  self._viewer_tool.ui, self._logger_tool.ui, self._correlator_tool.ui)
        self._e_log_printer = ELogPrinter(self._datalyzer_tool_ui, printers=elogs)
        self._timer_service = Timer(self._process_loop)
        self._io_service = io_service
        self._export_service = ExportService(application_name=application_name, export_directory=export_directory)
        self._create_connections()

    def _create_connections(self):
        """Helper method to create the required GUI connections."""
        self._create_datalyzer_tool_connections()
        self._create_viewer_tool_connections()
        self._create_logger_tool_connections()
        self._create_correlator_tool_connections()

    def _create_datalyzer_tool_connections(self):
        """Helper method to create the required datalyzer tool connections."""
        self._datalyzer_tool_ui.print_to_e_log_action.triggered.connect(self._e_log_printer.show_dialog)
        self._datalyzer_tool_ui.export_action.triggered.connect(self._export)

    def _create_viewer_tool_connections(self):
        """Helper method to create the required viewer tool connections."""
        self._create_tool_connections(self._viewer_tool)
        self._viewer_tool.ui.take_button.clicked.connect(self._initialize_background_taking_and_start_process_loop)

    def _create_logger_tool_connections(self):
        """Helper method to create the required logger tool connections."""
        self._create_tool_connections(self._logger_tool)

    def _create_correlator_tool_connections(self):
        """Helper method to create the required correlator tool connections."""
        self._create_tool_connections(self._correlator_tool)

    def _create_tool_connections(self, tool):
        """Helper method to create the tool connections."""
        tool.ui.start_button.clicked.connect(self._initialize_data_and_start_process_loop)
        tool.ui.stop_button.clicked.connect(self._stop_process_loop)

    def _initialize_data_and_start_process_loop(self):
        """Helper method to initialize the data and to start the process loop."""
        if self._logger_tool.is_started or self._correlator_tool.is_started:
            self._io_service.use_scalar_data(True)
        if self._viewer_tool.is_started:
            self._io_service.use_image_data(True)
            self._io_service.use_vector_data(True)
            self._io_service.use_dual_vector_data(True)
        self._start_process_loop()

    def _start_process_loop(self):
        """Helper method to start the process loop."""
        if not self._timer_service.is_active():
            self._io_service.initialize()
            self._timer_service.start()

    def _stop_process_loop(self):
        """Helper method to stop the process loop."""
        if not self._viewer_tool.is_started:
            self._io_service.use_image_data(False)
            self._io_service.use_vector_data(False)
            self._io_service.use_dual_vector_data(False)

        if not (self._logger_tool.is_started or self._correlator_tool.is_started):
            self._io_service.use_scalar_data(False)

        if self._timer_service.is_active() and self._all_tools_stopped():
            self._timer_service.stop()

    def _initialize_background_taking_and_start_process_loop(self):
        """Helper method to initialize the background taking and to start the process loop."""
        if self._viewer_tool.is_started:
            self._io_service.use_image_data(True)
            self._io_service.use_vector_data(True)
            self._io_service.use_dual_vector_data(True)
            self._start_process_loop()  # only when 'viewer tool' is started and not stopped due to invalid input

    def _all_tools_stopped(self):
        """Helper method to check if all tools have stopped."""
        return not (self._viewer_tool.is_started or self._logger_tool.is_started or self._correlator_tool.is_started)

    def _process_loop(self):
        """Helper method for looping the relevant processes."""
        self._update_io_service()
        self._update_tools()

    def _update_io_service(self):
        """Helper method to update the io services."""
        try:
            self._io_service.update()
        except (self._io_service.IOException,
                self._io_service.SynchronizationTimeout,
                self._io_service.SetDataException,
                self._io_service.SizeChangedWhileTakingBackground,
                self._io_service.SizeChangedWhileAveraging,
                self._io_service.SizeChangedWhileRunning) as exc:
            self._viewer_tool.ui.stop_button.click()
            self._logger_tool.ui.stop_button.click()
            self._correlator_tool.ui.stop_button.click()
            if isinstance(exc, (self._io_service.SizeChangedWhileAveraging, self._io_service.SizeChangedWhileRunning)):
                self._viewer_tool.ui.averaging_check_box.setChecked(False)
                self._viewer_tool.set_apply_background_correction_check_box_to(False)
            messages.show_error(str(exc))

    def _update_tools(self):
        """Helper method to update the tools."""
        if self._io_service.is_new_data_available:
            current_ui = self._datalyzer_tool_ui.get_current_ui()
            self._viewer_tool.update(to_plot=current_ui == self._viewer_tool.ui)
            self._logger_tool.update(to_plot=current_ui == self._logger_tool.ui)
            self._correlator_tool.update(to_plot=current_ui == self._correlator_tool.ui)

    def show_window(self):
        """Showing the PyQt GUI of the datalyzer application.

        Returns:
            None
        """
        self._datalyzer_tool_ui.show_window()

    def _export(self):
        """Helper method to export data to a file."""
        current_ui = self._datalyzer_tool_ui.get_current_ui()
        if current_ui == self._viewer_tool.ui:
            self._export_service.export(current_ui.name, self._viewer_tool.get_export_content())
        elif current_ui == self._logger_tool.ui:
            self._export_service.export(current_ui.name, self._logger_tool.get_export_content())
        elif current_ui == self._correlator_tool.ui:
            self._export_service.export(current_ui.name, self._correlator_tool.get_export_content())
        else:
            messages.show_error(f"Not available for '{current_ui.name}' tool")
