"""Module with correlator tool ui class for correlating data.

This module provides the correlator tool ui class for providing a tool for correlating data.
"""
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import (QMainWindow, QVBoxLayout, QWidget, QPushButton, QSizePolicy, QHBoxLayout, QGroupBox,
                             QSlider, QComboBox, QLabel)

from ..plotter.correlator_plotter import CorrelatorPlotter


class CorrelatorToolUi(QMainWindow):
    """Correlator tool class for correlating data.

    This class provides the ui for providing a tool for correlating data.
    """

    START_LABEL = "Start"

    def __init__(self, sources):
        """Constructor of the correlator tool ui class.

        This constructor initializes the correlator tool ui class.

        Args:
            sources (tuple): The data sources to visualize.
        """
        super().__init__()
        self._name = "Correlator"
        self._correlator_plotter = CorrelatorPlotter()
        self._main_layout = QVBoxLayout()
        self._start_button = self._get_labeled_push_button(self.START_LABEL)
        self._start_button.setEnabled(bool(sources))
        self._stop_button = self._get_labeled_push_button("Stop")
        self._reset_button = self._get_labeled_push_button("Reset")
        self._size_slider = QSlider(Qt.Horizontal)
        self._size_slider.setRange(5, 25)
        self._size_slider.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Preferred)
        self._alpha_slider = QSlider(Qt.Horizontal)
        self._alpha_slider.setRange(50, 255)
        self._alpha_slider.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Preferred)
        self._source_1_combo_box = self._get_combo_box_with(sources)
        self._source_2_combo_box = self._get_combo_box_with(sources)
        self._create_layout()
        self._create_connections()
        self._size_slider.setSliderPosition(10)
        self._alpha_slider.setSliderPosition(200)

    @staticmethod
    def _get_labeled_push_button(label):
        """Helper method to return a labeled push button."""
        push_button = QPushButton(label)
        push_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        return push_button

    @staticmethod
    def _get_combo_box_with(sources):
        """Helper method to return a customized combo box."""
        combo_box = QComboBox()
        combo_box.addItems(sources)
        return combo_box

    def _create_layout(self):
        """Helper method to invoke the multiple layout creators."""
        self._create_main_layout()
        self._create_plot_layout()
        self._create_controls_layout()

    def _create_main_layout(self):
        """Helper method to create the correlator tool's main layout."""
        main_widget = QWidget()
        main_widget.setLayout(self._main_layout)
        self.setCentralWidget(main_widget)

    def _create_plot_layout(self):
        """Helper method to create the correlator tool's plot layout."""
        plot_layout = QHBoxLayout()
        plot_layout.addWidget(self._correlator_plotter.widget)
        self._main_layout.addLayout(plot_layout)

    def _create_controls_layout(self):
        """Helper method to create the correlator tool's control layout."""
        controls_layout = QHBoxLayout()
        controls_layout.addWidget(self._create_control_group_box())
        controls_layout.addWidget(self._create_plot_symbols_group_box())
        controls_layout.addWidget(self._create_sources_group_box())
        self._main_layout.addLayout(controls_layout)

    def _create_control_group_box(self):
        """Helper method to create the control group's layout."""
        group_box = QGroupBox("Control")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._start_button)
        widgets_layout.addWidget(self._stop_button)
        widgets_layout.addWidget(self._reset_button)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_plot_symbols_group_box(self):
        """Helper method to create the plot symbols group's layout."""
        group_box = QGroupBox("Plot Symbols")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(QLabel("Size:"))
        widgets_layout.addWidget(self._size_slider)
        widgets_layout.addWidget(QLabel("Alpha:"))
        widgets_layout.addWidget(self._alpha_slider)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_sources_group_box(self):
        """Helper method to create the sources group's layout."""
        group_box = QGroupBox("Sources")
        widgets_layout = QHBoxLayout()
        widgets_layout.addWidget(self._source_1_combo_box)
        widgets_layout.addWidget(self._source_2_combo_box)
        group_box.setLayout(widgets_layout)
        return group_box

    def _create_connections(self):
        """Helper method to create connections."""
        self._size_slider.valueChanged.connect(self._update_size)
        self._alpha_slider.valueChanged.connect(self._update_alpha)

    @property
    def name(self):
        """str: The correlator tool's name."""
        return self._name

    @property
    def start_button(self):
        """QPushButton: The start button of the correlator tool."""
        return self._start_button

    @property
    def stop_button(self):
        """QPushButton: The stop button of the correlator tool."""
        return self._stop_button

    @property
    def reset_button(self):
        """QPushButton: The reset button of the correlator tool."""
        return self._reset_button

    @property
    def source_1_combo_box(self):
        """QComboBox: The source 1 combo box of the correlator tool."""
        return self._source_1_combo_box

    @property
    def source_2_combo_box(self):
        """QComboBox: The source 2 combo box of the correlator tool."""
        return self._source_2_combo_box

    def plot(self, data_1, data_2, axis_labels):
        """Plotting the given data with axis labels.

        Args:
            data_1 (property dependent): The property dependent data-1 to correlate.
            data_2 (property dependent): The property dependent data-2 to correlate.
            axis_labels (tuple): The axis labels for the data.

        Returns:
            None
        """
        self._correlator_plotter.set(data_1, data_2, axis_labels)

    def _update_size(self, size):
        """Helper method to update the symbol's size"""
        self._correlator_plotter.update_size(size)

    def _update_alpha(self, alpha):
        """Helper method to update the symbol's size"""
        self._correlator_plotter.update_alpha(alpha)

    def reset_plot_range(self):
        """Resetting the plot range.

        Returns:
            None
        """
        self._correlator_plotter.reset_plot_range()
