"""Module with correlator tool class for correlating data.

This module provides the correlator tool class for providing a tool for correlating data.
"""
from collections import defaultdict

from .correlator_tool_ui import CorrelatorToolUi


class CorrelatorTool:
    """Correlator tool class for correlating data.

    This class provides the methods for providing a tool for correlating data.
    """

    def __init__(self, scalar_data, axis_labels):
        """Constructor of the correlator tool class.

        This constructor initializes the correlator tool class.

        Args:
            scalar_data (ScalarData): The scalar data of the readout sources.
            axis_labels (dict): The axis labels for the data sources.
        """
        self._correlator_tool_ui = CorrelatorToolUi(scalar_data.get_sources())
        self._scalar_data = scalar_data
        self._axis_labels = axis_labels
        self._actual_data = defaultdict(list)
        self._selected_sources = None
        self._is_started = False
        self._create_connections()
        self._select_sources_to_plot()

    def _create_connections(self):
        """Helper method to create the required GUI connections."""
        self._correlator_tool_ui.start_button.clicked.connect(self._start_button_clicked)
        self._correlator_tool_ui.stop_button.clicked.connect(self._stop_button_clicked)
        self._correlator_tool_ui.reset_button.clicked.connect(self._reset_button_clicked)
        self._correlator_tool_ui.source_1_combo_box.currentIndexChanged.connect(self._select_sources_to_plot)
        self._correlator_tool_ui.source_1_combo_box.currentIndexChanged.connect(self._offline_update)
        self._correlator_tool_ui.source_1_combo_box.currentIndexChanged.connect(
            self._correlator_tool_ui.reset_plot_range)
        self._correlator_tool_ui.source_2_combo_box.currentIndexChanged.connect(self._select_sources_to_plot)
        self._correlator_tool_ui.source_2_combo_box.currentIndexChanged.connect(self._offline_update)
        self._correlator_tool_ui.source_2_combo_box.currentIndexChanged.connect(
            self._correlator_tool_ui.reset_plot_range)

    @property
    def ui(self):
        """CorrelatorToolUi: The correlator tool user interface."""
        return self._correlator_tool_ui

    @property
    def is_started(self):
        """bool: The running state of the GUI."""
        return self._is_started

    def _start_button_clicked(self):
        """Helper method to realize the actions for clicking the 'start' button."""
        self._is_started = True
        self._correlator_tool_ui.start_button.setText("running ...")
        self._correlator_tool_ui.start_button.setEnabled(False)

    def _stop_button_clicked(self):
        """Helper method to realize the actions for clicking the 'stop' button."""
        if self._is_started:
            self._correlator_tool_ui.start_button.setText(self._correlator_tool_ui.START_LABEL)
            self._correlator_tool_ui.start_button.setEnabled(True)
            self._is_started = False

    def _reset_button_clicked(self):
        """Helper method to realize the actions for clicking the 'reset' button."""
        if (not self._is_started and self._actual_data[self._selected_sources[0]] and
                self._actual_data[self._selected_sources[1]]):
            self._correlator_tool_ui.plot([], [], ("", ""))
        self._actual_data = defaultdict(list)

    def _select_sources_to_plot(self):
        """Helper method to select the source to plot."""
        self._selected_sources = (self._correlator_tool_ui.source_1_combo_box.currentText(),
                                  self._correlator_tool_ui.source_2_combo_box.currentText())

    def update(self, to_plot):
        """Updating the correlator tool.

        Returns:
            None
        """
        if self._is_started:
            self._append_actual_data()
            if to_plot:
                self._plot()

    def _append_actual_data(self):
        """Helper method to append actual data."""
        for source in self._scalar_data.get_sources():
            self._actual_data[source].append(self._scalar_data.get(source))

    def _offline_update(self):
        """Helper method to offline update the correlator tool."""
        if (not self._is_started and self._actual_data[self._selected_sources[0]] and
                self._actual_data[self._selected_sources[1]]):
            self._plot()

    def _plot(self):
        """Helper method to plot the selected data with given labels."""
        self._correlator_tool_ui.plot(self._actual_data[self._selected_sources[0]],
                                      self._actual_data[self._selected_sources[1]],
                                      (self._axis_labels.get(self._selected_sources[0], (None, None))[1],
                                       self._axis_labels.get(self._selected_sources[1], (None, None))[1]))

    def get_export_content(self):
        """Getting the content to export to a file.

        Returns:
            dict
        """
        return {"selected source 1": [self._selected_sources[0]],
                "selected source 2": [self._selected_sources[1]],
                "axis label 1": [self._axis_labels.get(self._selected_sources[0], (None, None))[1]],
                "axis label 2": [self._axis_labels.get(self._selected_sources[1], (None, None))[1]],
                "source data 1": self._actual_data[self._selected_sources[0]],
                "source data 2": self._actual_data[self._selected_sources[1]]}
