"""Module with io service class for I/O within DOOCS.

This module provides the io service class for I/O operations with the DOOCS control system.
"""
import time

import numpy
import numpy as np

import doocspie
from doocspie.abo.train_event import TrainEvent
from doocspie.gui.datalyzer.materials.base_data import BaseData


class IOService:
    """IO service class for I/O within DOOCS.

    This class provides the methods for I/O operations with the DOOCS control system.
    """

    # custom exceptions from BaseData class for reuse
    SizeChangedWhileTakingBackground = BaseData.SizeChangedWhileTakingBackground
    SizeChangedWhileRunning = BaseData.SizeChangedWhileRunning
    SizeChangedWhileAveraging = BaseData.SizeChangedWhileAveraging

    class IOException(Exception):
        """Helper class to provide a dedicated IO exception."""

        def __init__(self, error):
            """Constructor of the io exception class.

            This constructor initializes the io exception class.

            Args:
                error (str): The causing exception as string representation.
            """
            super().__init__(f"I/O error: {error}")

    class SetDataException(Exception):
        """Helper class to provide a dedicated exception related to setting data."""

        def __init__(self, error):
            """Constructor of the exception class related to setting data.

            This constructor initializes the exception class related to setting data.

            Args:
                error (str): The causing exception as string representation.
            """
            super().__init__(f"Set data error: {error}")

    class SynchronizationTimeout(Exception):
        """Helper class to provide a dedicated synchronization timeout exception."""

        def __init__(self, timeout_seconds):
            """Constructor of the synchronization timeout exception class.

            This constructor initializes the synchronization timeout exception class.

            Args:
                timeout_seconds (int): The timeout seconds for unsuccessful readout.
            """
            super().__init__(f"Synchronization timeout: {timeout_seconds} seconds")

    def __init__(self, synchronizer, registered_functions, image_data, vector_data, dual_vector_data, scalar_data):
        """Constructor of the io service class.

        This constructor initializes the io service class for I/O within DOOCS.

        Args:
            synchronizer (Synchronizer): The synchronizer for synchronous readout.
            registered_functions (dict): Functions to provide custom data metrics.
            image_data (ImageData): The image data of the readout sources.
            vector_data (VectorData): The vector data of the readout sources.
            dual_vector_data (DualVectorData): The dual vector data of the readout sources.
            scalar_data (ScalarData): The scalar data of the readout sources.
        """
        self._synchronizer = synchronizer
        self._registered_functions = registered_functions
        self._image_data = image_data
        self._vector_data = vector_data
        self._dual_vector_data = dual_vector_data
        self._scalar_data = scalar_data
        self._use_image_data = False
        self._use_vector_data = False
        self._use_dual_vector_data = False
        self._use_scalar_data = False
        self._is_new_data_available = None
        self._last_time = None
        self._initialize_data_sources()

    @property
    def image_data(self):
        """ImageData: The image data of the readout sources."""
        return self._image_data

    @property
    def vector_data(self):
        """VectorData: The vector data of the readout sources."""
        return self._vector_data

    @property
    def dual_vector_data(self):
        """DualVectorData: The dual vector data of the readout sources."""
        return self._dual_vector_data

    @property
    def scalar_data(self):
        """ScalarData: The scalar data of the readout sources."""
        return self._scalar_data

    @property
    def is_new_data_available(self):
        """bool: The state of new data availability."""
        return self._is_new_data_available

    def _initialize_data_sources(self):
        """Helper method to check the data sources."""
        self._add_data_source_properties()
        self._add_data_source_functions()

    def _add_data_source_properties(self):
        """Helper methods to add the data source properties."""
        for property in self._synchronizer.usable_properties + self._synchronizer.zero_event_properties:
            try:
                readout = doocspie.get(property.address)
            except doocspie.DoocspieException:
                continue

            if isinstance(readout.data, numpy.ndarray):
                if readout.data.ndim == 2:
                    self._image_data.add_property(property)
                elif readout.data.ndim == 1:
                    self._vector_data.add_property(property)
                else:
                    self._raise_data_type_not_supported(readout, property)
            elif isinstance(readout.data, (int, float)):
                self._scalar_data.add_property(property)
            else:
                self._raise_data_type_not_supported(readout, property)

    @staticmethod
    def _raise_data_type_not_supported(readout, property):
        """Helper method to raise an exception for not supported data types."""
        doocs_property = property.address.split("/")[-1]
        raise doocspie.DoocspieException(f"data type '{readout.type}' of '{doocs_property}' not supported",
                                         property.address)

    def _add_data_source_functions(self):
        """Helper methods to add the data source functions."""
        train_event = TrainEvent(self._synchronizer.get(), self._synchronizer.actual_properties)
        for name, function in self._registered_functions.items():
            try:
                readout_data = function(train_event)
            except doocspie.DoocspieException:
                continue

            if isinstance(readout_data, numpy.ndarray):
                if readout_data.ndim == 2:
                    self._image_data.add_function(name, function)
                elif readout_data.ndim == 1:
                    self._vector_data.add_function(name, function)
                else:
                    raise doocspie.DoocspieException(f"returned array dimension '{readout_data.ndim}' of registered "
                                                     f"'{name}' function not supported")
            elif isinstance(readout_data, (int, float, np.integer, np.floating)):
                self._scalar_data.add_function(name, function)
            elif isinstance(readout_data, (tuple, list)):
                self._add_dual_vector_functions(readout_data, name, function)
            else:
                raise doocspie.DoocspieException(f"return type '{type(readout_data).__name__}' of registered '{name}' "
                                                 f"function not supported")

    def _add_dual_vector_functions(self, readout_data, name, function):
        """Helper methods to add dual vector function."""
        if isinstance(readout_data[0], numpy.ndarray) and isinstance(readout_data[1], numpy.ndarray):
            if readout_data[0].ndim == readout_data[1].ndim == 1:
                if len(readout_data[0]) == len(readout_data[1]):
                    self._dual_vector_data.add_function(name, function)
                else:
                    raise doocspie.DoocspieException(f"returned array lengths '{len(readout_data[0])}' and "
                                                     f"'{len(readout_data[1])}' of registered '{name}' function do not "
                                                     f"match")
            else:
                raise doocspie.DoocspieException(f"returned array dimensions '{readout_data[0].ndim}' and "
                                                 f"'{readout_data[1].ndim}' of registered '{name}' function not "
                                                 f"supported")
        else:
            raise doocspie.DoocspieException(f"return types '{type(readout_data[0]).__name__}' and "
                                             f"'{type(readout_data[1]).__name__}' of registered '{name}' function not "
                                             f"supported")

    def _get_readouts(self):
        """Helper method to get the readouts with additional exception passing."""
        try:
            return self._synchronizer.get_event_matched_readouts()
        except Exception as exc:
            raise self.IOException(str(exc))

    def initialize(self):
        """Initializing the io service.

        Returns:
            None
        """
        self._last_time = time.time()

    def use_image_data(self, state):
        """Set the usage state for image data.

        Args:
            state (bool): The usage state for image data.

        Returns:
            None
        """
        self._use_image_data = state

    def use_vector_data(self, state):
        """Set the usage state for vector data.

        Args:
            state (bool): The usage state for vector data.

        Returns:
            None
        """
        self._use_vector_data = state

    def use_dual_vector_data(self, state):
        """Set the usage state for dual vector data.

        Args:
            state (bool): The usage state for dual vector data.

        Returns:
            None
        """
        self._use_dual_vector_data = state

    def use_scalar_data(self, state):
        """Set the usage state for scalar data.

        Args:
            state (bool): The usage state for scalar data.

        Returns:
            None
        """
        self._use_scalar_data = state

    def update(self):
        """Updating the io service.

        Returns:
            None
        """
        readouts = self._get_readouts()
        self._is_new_data_available = False
        if readouts:
            train_event = TrainEvent(readouts, self._synchronizer.actual_properties)
            self._set_data(train_event)
            self._is_new_data_available = True
            self._last_time = time.time()

        if (time.time() - self._last_time) > self._synchronizer.timeout_seconds:
            raise self.SynchronizationTimeout(self._synchronizer.timeout_seconds)

    def _set_data(self, train_event):
        """Helper method to set all relevant data."""
        try:
            if self._use_image_data:
                self._image_data.set(train_event)
            if self._use_vector_data:
                self._vector_data.set(train_event)
            if self._use_dual_vector_data:
                self._dual_vector_data.set(train_event)
            if self._use_scalar_data:
                self._scalar_data.set(train_event)
        except self.SizeChangedWhileTakingBackground as exc:
            raise exc
        except self.SizeChangedWhileRunning as exc:
            raise exc
        except self.SizeChangedWhileAveraging as exc:
            raise exc
        except Exception as exc:
            raise self.SetDataException(str(exc))
