"""Module with viewer plotter class for plotting viewer data.

This module provides the plotter class for providing a plotter for viewer data.
"""
import numpy as np
from doocspie.util import Colors
from pyqtgraph import setConfigOptions, GraphicsLayoutWidget, ImageItem, PlotDataItem, HistogramLUTItem


class ViewerPlotter:
    """Viewer plotter class for plotting viewer data.

    This class provides the methods for plotting viewer data.
    """
    setConfigOptions(imageAxisOrder="row-major")

    _DESY_CYAN = np.array(Colors.DESY.CYAN) * 255

    IMAGE = "image"
    VECTOR = "vector"
    DUAL_VECTOR = "dual vector"

    def __init__(self):
        """Constructor of the viewer plotter class.

        This constructor initializes the viewer plotter class.
        """
        self._widget = GraphicsLayoutWidget()
        self._plot = self._widget.addPlot()
        self._image = ImageItem()
        self._histogram = None
        self._histogram_added = False
        self._any_vector = PlotDataItem(pen={"color": self._DESY_CYAN, "width": 2})
        self._current_plot_type = None
        self._first_readout = None

    @property
    def widget(self):
        """GraphicsLayoutWidget: The viewer plotter widget."""
        return self._widget

    def set_plot_type(self, plot_type):
        """Setting the plot type of actual viewer plotter.

        Args:
            plot_type (str): The plot type (either IMAGE or VECTOR) to set.

        Returns:
            None
        """
        if plot_type == self.IMAGE and self._current_plot_type != self.IMAGE:
            self._plot.removeItem(self._any_vector)
            self._plot.addItem(self._image)
            self._histogram = HistogramLUTItem()
            self._histogram.setImageItem(self._image)
            self._widget.addItem(self._histogram)
            self._histogram_added = True
            self._plot.invertY(True)
            self._first_readout = True
        elif (plot_type in (self.VECTOR, self.DUAL_VECTOR) and
              self._current_plot_type not in (self.VECTOR, self.DUAL_VECTOR)):
            self._plot.removeItem(self._image)
            if self._histogram_added:
                self._widget.removeItem(self._histogram)
                self._histogram_added = False
            self._plot.addItem(self._any_vector)
            self._plot.invertY(False)
        self._current_plot_type = plot_type
        self._initialize()

    def _initialize(self):
        """Helper method to initialize the plot."""
        self._plot.setMenuEnabled(False)
        self._plot.showAxis("top", show=True)
        self._plot.showAxis("right", show=True)
        for position in "top", "right", "left", "bottom":
            self._plot.getAxis(position).setStyle(showValues=False, tickLength=0)

    def set(self, data, axis_labels):
        """Setting the given data with axis labels in the viewer plotter.

        Args:
            data (property dependent): The property dependent data to set in the viewer plotter.
            axis_labels (tuple): The axis labels for the data.

        Returns:
            None
        """
        if self._current_plot_type == self.IMAGE:
            self._image.setImage(data, autoLevels=self._first_readout)
            self._first_readout = False
        elif self._current_plot_type == self.VECTOR:
            self._any_vector.setData(data)
        elif self._current_plot_type == self.DUAL_VECTOR:
            self._any_vector.setData(data[0], data[1])
        self._plot.getAxis("bottom").setStyle(showValues=True)
        self._plot.setLabel("bottom", axis_labels[0])
        self._plot.getAxis("left").setStyle(showValues=True)
        self._plot.setLabel("left", axis_labels[1])

    def reset_plot_range(self):
        """Resetting the plot range.

        Returns:
            None
        """
        self._plot.enableAutoRange()
        self._first_readout = True
