"""Module with logger plotter class for plotting logger data.

This module provides the plotter class for providing a plotter for logger data.
"""
import numpy as np
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QGraphicsLineItem
from doocspie.util import Colors
from pyqtgraph import GraphicsLayoutWidget, PlotDataItem, mkPen, DateAxisItem, AxisItem


class LoggerPlotter:
    """Logger plotter class for plotting logger data.

    This class provides the methods for plotting logger data.
    """
    _DESY_CYAN = np.array(Colors.DESY.CYAN) * 255
    _DESY_ORANGE = np.array(Colors.DESY.ORANGE) * 255

    class _MovableLine(QGraphicsLineItem):
        """Helper class to provide a movable line."""

        def __init__(self, plot, color, width):
            super().__init__()
            self._plot = plot
            self.setPen(mkPen(color=color, width=width))
            self.setAcceptHoverEvents(True)
            self._saved_pen = None
            self._delta_when_pressed = None

        def hoverEnterEvent(self, event):
            self._saved_pen = self.pen()
            self.setPen(mkPen(color=self._saved_pen.color().getRgb(), width=2 * self._saved_pen.width()))
            event.ignore()

        def hoverLeaveEvent(self, event):
            self.setPen(self._saved_pen)
            event.ignore()

        def mousePressEvent(self, event):
            if event.button() == Qt.LeftButton:
                event.accept()
                self._plot.enableAutoRange(axis="y", enable=False)
                self._delta_when_pressed = self.mapToParent(event.pos()) - self.pos()
            else:
                event.ignore()

        def mouseReleaseEvent(self, event):
            self._plot.enableAutoRange(axis="y", enable=True)
            event.ignore()

        def mouseMoveEvent(self, event):
            self.setPos(0, (self.mapToParent(event.pos()) - self._delta_when_pressed).y())

    def __init__(self, sources):
        """Constructor of the logger plotter class.

        This constructor initializes the logger plotter class.

        Args:
            sources (tuple): The data sources for the movable line.
        """
        self._widget = GraphicsLayoutWidget()
        self._plot = self._widget.addPlot()
        self._scalar = PlotDataItem(pen={"color": self._DESY_CYAN, "width": 2})
        self._plot.addItem(self._scalar)
        self._movable_line_position = {source: 0 for source in sources}
        self._movable_line = self._MovableLine(self._plot, color=self._DESY_ORANGE, width=3)
        self._plot.addItem(self._movable_line, ignoreBounds=True)
        self._initialize()
        self._use_timestamps = None
        self._is_movable_line_visible = False
        self._movable_line_position_value = None

    def _initialize(self):
        """Helper method to initialize the plot."""
        self._plot.setMenuEnabled(False)
        self._plot.showAxis("top", show=True)
        self._plot.showAxis("right", show=True)
        for position in "top", "right", "left", "bottom":
            self._plot.getAxis(position).setStyle(showValues=False, tickLength=0)

    @property
    def widget(self):
        """GraphicsLayoutWidget: The logger plotter widget."""
        return self._widget

    @property
    def movable_line_position_value(self):
        """Float: The position value of the movable line."""
        if self._is_movable_line_visible:
            return self._movable_line_position_value
        return None

    def set_movable_line(self, selected_source):
        """Setting the source for the movable line.

        Args:
            selected_source (str): The selected data source for the movable line.

        Returns:
            None
        """
        self._movable_line.setY(self._movable_line_position[selected_source])

    def set(self, data, axis_label, selected_source):
        """Setting the given data with axis labels in the logger plotter, and showing a movable line.

        Args:
            data (tuple): The data to set in the logger plotter.
            axis_label (str): The axis label for the data.
            selected_source (str): The selected data source for the movable line.

        Returns:
            None
        """
        if self._use_timestamps:
            self._scalar.setData(x=data[0], y=data[1], stepMode="right")
            self._plot.setLabel("bottom", "Timestamp")
        else:
            self._scalar.setData(x=data[0], y=data[1], stepMode=None)
            self._plot.setLabel("bottom", "Sample")
        self._plot.getAxis("bottom").setStyle(showValues=True)
        self._plot.getAxis("left").setStyle(showValues=True)
        self._plot.setLabel("left", axis_label)
        self._set_movable_line(data, selected_source)

    def _set_movable_line(self, data, selected_source):
        """Helper method to set the data of the movable line."""
        finite_data = np.array(data[1])[np.isfinite(data[1])]
        if finite_data.size:
            value = finite_data[0]
            self._movable_line_position[selected_source] = self._movable_line.pos().y()
            self._movable_line.setLine(data[0][0], value, data[0][-1], value)
            self._movable_line.setVisible(self._is_movable_line_visible)
            self._movable_line_position_value = float(value + self._movable_line.pos().y())

    def reset_plot_range(self):
        """Resetting the plot range.

        Returns:
            None
        """
        self._plot.enableAutoRange()

    def reset(self):
        """Resetting the movable line.

        Returns:
            None
        """
        self._movable_line.setVisible(False)
        self._movable_line.setY(0)
        for key in self._movable_line_position:
            self._movable_line_position[key] = 0
        self._movable_line_position_value = None

    def set_movable_line_visible(self, state):
        """Setting the movable line's visibility.

        Args:
            state (bool): The state of the movable line's visibility.

        Returns:
            None
        """
        self._is_movable_line_visible = state
        if self._scalar.getData()[0] is not None and any(self._scalar.getData()[0]):
            self._movable_line.setVisible(state)

    def use_timestamps(self, state):
        """Using the timestamps according to the state.

        Args:
            state (bool): The usage state of timestamps.

        Returns:
            None
        """
        self._use_timestamps = state
        if self._scalar.getData()[0] is not None:
            if state:
                self._plot.setAxisItems({"bottom": DateAxisItem()})
            else:
                self._plot.setAxisItems({"bottom": AxisItem(orientation="bottom")})
        self._plot.getAxis("bottom").setStyle(tickLength=0)
