"""Module with correlator plotter class for plotting correlator data.

This module provides the plotter class for providing a plotter for correlator data.
"""
import numpy as np
from doocspie.util import Colors
from pyqtgraph import GraphicsLayoutWidget, PlotDataItem


class CorrelatorPlotter:
    """Correlator plotter class for plotting correlator data.

    This class provides the methods for plotting correlator data.
    """
    _DESY_ORANGE = np.array(Colors.DESY.ORANGE) * 255

    def __init__(self):
        """Constructor of the correlator plotter class.

        This constructor initializes the correlator plotter class.
        """
        self._widget = GraphicsLayoutWidget()
        self._plot = self._widget.addPlot()
        self._correlation = PlotDataItem(symbol="o", pen=None, symbolPen=None)
        self._plot.addItem(self._correlation)
        self._initialize()

    def _initialize(self):
        """Helper method to initialize the plot."""
        self._plot.setMenuEnabled(False)
        self._plot.showAxis("top", show=True)
        self._plot.showAxis("right", show=True)
        for position in "top", "right", "left", "bottom":
            self._plot.getAxis(position).setStyle(showValues=False, tickLength=0)

    @property
    def widget(self):
        """GraphicsLayoutWidget: The correlator plotter widget."""
        return self._widget

    def set(self, data_1, data_2, axis_labels):
        """Setting the given data with axis labels in the correlator plotter.

        Args:
            data_1 (property dependent): The property dependent data-1 to set in the correlator plotter.
            data_2 (property dependent): The property dependent data-2 to set in the correlator plotter.
            axis_labels (tuple): The axis labels for the data.

        Returns:
            None
        """
        self._correlation.setData(x=data_1, y=data_2)
        self._plot.getAxis("bottom").setStyle(showValues=True)
        self._plot.setLabel("bottom", axis_labels[0])
        self._plot.getAxis("left").setStyle(showValues=True)
        self._plot.setLabel("left", axis_labels[1])

    def update_size(self, size):
        """Updating the symbol's size.

        Args:
            size (int): The symbol size of the plot points.

        Returns:
            None
        """
        self._correlation.setSymbolSize(size)

    def update_alpha(self, alpha):
        """Updating the symbol's alpha.

        Args:
            alpha (int): The symbol alpha channel of the plot points.

        Returns:
            None
        """
        self._correlation.setSymbolBrush(np.append(self._DESY_ORANGE, alpha))

    def reset_plot_range(self):
        """Resetting the plot range.

        Returns:
            None
        """
        self._plot.enableAutoRange()
