"""Module with base data class for dependent actual data classes.

This module offers the base data class for dependent actual data classes.
"""
from collections import defaultdict
import copy

import numpy as np


class BaseData:
    """Base data class for dependent actual data classes.

    This class offers the base data class for dependent actual data classes.
    """

    class SizeChangedWhileTakingBackground(Exception):
        """Helper class to provide a dedicated exception related to changed size while taking background."""

        def __init__(self, property):
            """Constructor of the exception class related to changed size while taking background.

            This constructor initializes the exception class related to changed size while taking background.

            Args:
                property (str): The exception causing property.
            """
            super().__init__(f"Data size changed while taking background for '{property}'")

    class SizeChangedWhileRunning(Exception):
        """Helper class to provide a dedicated exception related to changed size while running."""

        def __init__(self, property):
            """Constructor of the exception class related to changed size while running.

            This constructor initializes the exception class related to changed size while running.

            Args:
                property (str): The exception causing property.
            """
            super().__init__(f"Data size changed while running for '{property}'")

    class SizeChangedWhileAveraging(Exception):
        """Helper class to provide a dedicated exception related to changed size while averaging."""

        def __init__(self, property):
            """Constructor of the exception class related to changed size while averaging.

            This constructor initializes the exception class related to changed size while averaging.

            Args:
                property (str): The exception causing property.
            """
            super().__init__(f"Data size changed while averaging for '{property}'")

    class _SizeChanged(Exception):  # marker exception
        """Helper class to provide a generic marker exception related to changed size."""

    def __init__(self):
        """Constructor of the base data class.

        This constructor initializes the instance without any required parameters.
        """
        self._properties = []
        self._functions = []
        self._data = {}
        self._is_averaging = None
        self._average_samples = defaultdict(lambda: 0)
        self._averaged_data = defaultdict(lambda: None)
        self._is_taking_background = None
        self._is_correcting_background = None
        self._is_background_corrected = False
        self._background_samples = None
        self._background_sample = None
        self._background_data = defaultdict(lambda: None)
        self._corrected_data = defaultdict(lambda: None)

    @property
    def is_taking_background(self):
        """bool: State of background taking."""
        return self._is_taking_background

    @property
    def is_background_corrected(self):
        """bool: State of background correction."""
        return self._is_background_corrected

    def add_property(self, property):
        """Add data source property for synchronous readout.

        Args:
            property (namedtuple): The data source property to add for synchronous readout.

        Returns:
            None
        """
        if property.label:
            self._properties.append(property.label)
        else:
            self._properties.append(property.address)

    def add_function(self, name, function):
        """Add data source function for synchronous readout.

        Args:
            name (str): The name of the data source function to add for synchronous readout.
            function (function): The actual data source function to add for synchronous readout.

        Returns:
            None
        """
        self._functions.append((name, function))

    def get_sources(self):
        """Get the data sources to get readouts for.

        Returns:
            tuple: The data sources to get readouts for.
        """
        return tuple(self._properties + [function_name for function_name, _ in self._functions])

    def set_averaging(self, state):
        """Set averaging according to the given state.

        Args:
            state (bool): The averaging set state.

        Returns:
            None
        """
        self._is_averaging = state
        self._average_samples = defaultdict(lambda: 0)

    def set_background_taking(self, samples):
        """Set background taking with the given number of samples.

        Args:
            samples (int): The number of background samples.

        Returns:
            None
        """
        self._is_taking_background = True
        self._background_samples = samples
        self._background_sample = defaultdict(lambda: 0)

    def set_background_correction(self, state):
        """Set background correction to the given state.

        Args:
            state (bool): The averaging set state.

        Returns:
            None
        """
        self._is_correcting_background = state

    def reset_background_taking(self):
        """Reset background taking.

        Returns:
            None
        """
        self._is_taking_background = False

    def set(self, train_event):
        """Set all relevant data from the given train event.

        Args:
            train_event (TrainEvent): The train event to get the data to set from.

        Returns:
            None
        """
        self._set(train_event)
        self._is_background_corrected = False
        if self._is_taking_background:
            self._apply_background_taking()
        else:
            if self._is_correcting_background:
                self._apply_background_correction()
                self._is_background_corrected = True
            if self._is_averaging:
                self._apply_averaging()

    def _set(self, train_event):
        """Helper method to set the data from the given train event."""
        for property in self._properties:
            readout_data = train_event.get(property).data
            # type conversion below is required for bgr subtraction of integral types
            if isinstance(readout_data, np.ndarray):
                readout_data = readout_data.astype(np.float32)
            self._data[property] = readout_data

        for name, function in self._functions:
            try:
                readout_data = function(train_event)
                if isinstance(readout_data, tuple):
                    readout_data = list(readout_data)
                    readout_data[1] = readout_data[1].astype(np.float32)  # see type conversion comment above
                else:
                    if isinstance(readout_data, np.ndarray):
                        readout_data = readout_data.astype(np.float32)  # see type conversion comment above
                self._data[name] = copy.deepcopy(readout_data)
            except Exception as exc:
                raise Exception(str(exc) + " in '" + name + "'")

    def _apply_background_taking(self):
        """Helper method to apply background taking."""
        for property in self._data:
            if isinstance(self._data[property], list):
                data = self._data[property][1]
            else:
                data = self._data[property]

            try:
                self._background_sample[property], self._background_data[property] = self._averaging(
                    self._background_sample[property], self._background_data[property], data)
            except self._SizeChanged:
                raise self.SizeChangedWhileTakingBackground(property)
            finally:
                if isinstance(self._data[property], list):
                    self._data[property][1] = self._background_data[property]
                else:
                    self._data[property] = self._background_data[property]

    def _apply_background_correction(self):
        """Helper method to apply background correction."""
        for property in self._data:
            if isinstance(self._data[property], list):
                data = self._data[property][1]
            else:
                data = self._data[property]

            if data.shape == self._background_data[property].shape:
                self._corrected_data[property] = data - self._background_data[property]
                data[:] = self._corrected_data[property]
            else:
                if self._corrected_data[property] is None:
                    if isinstance(self._data[property], list):
                        self._data[property][1] = self._background_data[property]
                    else:
                        self._data[property] = self._background_data[property]
                else:
                    if isinstance(self._data[property], list):
                        self._data[property][1] = self._corrected_data[property]
                    else:
                        self._data[property] = self._corrected_data[property]
                raise self.SizeChangedWhileRunning(property)

    def _apply_averaging(self):
        """Helper method to apply averaging."""
        for property in self._data:
            if isinstance(self._data[property], list):
                data = self._data[property][1]
            else:
                data = self._data[property]

            try:
                self._average_samples[property], self._averaged_data[property] = self._averaging(
                    self._average_samples[property], self._averaged_data[property], data)
            except self._SizeChanged:
                raise self.SizeChangedWhileAveraging(property)
            finally:
                if isinstance(self._data[property], list):
                    self._data[property][1] = self._averaged_data[property]
                else:
                    self._data[property] = self._averaged_data[property]

    @classmethod
    def _averaging(cls, samples, average, data):
        """Helper method for averaging."""
        samples += 1
        if samples == 1:
            average = data
        elif data.shape == average.shape:
            inverse_samples = 1 / samples  # use this to avoid memory issue due to implicit casting
            average = (1 - inverse_samples) * average + inverse_samples * data
        else:
            raise cls._SizeChanged()
        return samples, average

    def get(self, source):
        """Get the readout data of the given source.

        Args:
            source (str): The source to get the data from.

        Returns:
            property dependent: The readout data of the given source.
        """
        return self._data.get(source)

    def get_average_samples(self, source):
        """Get the average samples of the given source.

        Args:
            source (str): The source to get the average samples for.

        Returns:
            str: The average samples as a string.
        """
        return str(self._average_samples[source])

    def get_background_taking_progress(self, source):
        """Get the background taking progress of the given source.

        Args:
            source (str): The source to get the background taking progress for.

        Returns:
            int: The background taking progress.
        """
        return int(self._background_sample[source] / self._background_samples * 100)
