"""Module with datalyzer class for providing a scriptable graphical user interface (GUI).

This module provides the datalyzer class for providing a scriptable GUI for data visualization.
"""
import os
import signal
import sys

from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QApplication

from doocspie.abo.synchronizer import Synchronizer
from doocspie.doocspie import DoocspieException

from .materials.image_data import ImageData
from .materials.vector_data import VectorData
from .materials.dual_vector_data import DualVectorData
from .materials.scalar_data import ScalarData
from .services.io_service import IOService
from .tools.datalyzer_tool import DatalyzerTool

#: The GUI application's name
__application_name__ = "Datalyzer"

#: The GUI application's version number
__version__ = "1.0.0"

#: The GUI application's available e-logs to print to.
__elogs__ = {"FLASH": "ttflog", "FLASH1 Photon Diag.": "exphalllog", "FLASH2 Photon Diag.": "fl2_photlog",
             "FLASH1 CAMP": "fl1_camplog", "FLASH1 BL2": "fl1_bl2log", "FLASH1 BL3": "fl1_bl3log",
             "FLASH1 PG": "felpgmlog",
             "FLASH2 FL23": "fl2_fl23log", "FLASH2 FL24": "fl2_fl24log", "FLASH2 FL26": "fl2_fl26log",
             "FLASH1 Compact Spectrometer": "cpspectlog", "FLASH1 Raman": "fl1_ramlog", "FLASH1 THz": "fl1_thzlog",
             "FLASH1 THz Streaking": "fl_thzsrklog", "HexTOF": "hextoflog", "WESPE": "wespelog", "MUSIX": "musixlog",
             "FLASH1 Pump-Probe Laser": "pplaserlog", "FLASH2 Pump-Probe Laser": "fl2_pplaslog"}

#: The GUI application's file export directory.
__export_directory__ = os.path.expanduser("~") + "/"


class Datalyzer:
    """Datalyzer class for providing a scriptable graphical user interface (GUI).

    This class provides the methods for providing a scriptable GUI for data visualization.
    """

    def __init__(self, properties=None, timeout_seconds=10, buffer_size=32, allow_zero_events=False):
        """Constructor of the datalyzer class.

        This constructor initializes the instance with optional parameters (see 'Args' below).

        Args:
            properties (tuple or dict, optional): The optional properties to get train events with readouts for.
            timeout_seconds (int): The optional timeout seconds for unsuccessful readout.
            buffer_size (int): The optional number of buffers to be used for storing previous readouts.
            allow_zero_events (bool, optional): The optional state for allow zero train events.
        """
        self._synchronizer = Synchronizer(properties, timeout_seconds, buffer_size, allow_zero_events)
        self._registered_functions = {}
        self._axis_labels = {}
        self._synchronizer_to_update = False

    @property
    def registered_functions(self):
        """tuple: The registered functions that can be used for synchronized readout in the datalyzer."""
        return tuple(self._registered_functions.keys())

    @property
    def usable_properties(self):
        """tuple: The properties that can be used for synchronized readout in the datalyzer."""
        return self._synchronizer.usable_properties

    @property
    def unusable_properties(self):
        """tuple: The properties that cannot be used for synchronized readout in the datalyzer."""
        return self._synchronizer.unusable_properties

    @property
    def zero_event_properties(self):
        """tuple: The properties that have an event number of 'zero'."""
        return self._synchronizer.zero_event_properties

    @property
    def actual_properties(self):
        """tuple: The properties that can actually be readout in the datalyzer."""
        return self._synchronizer.actual_properties

    def add(self, address, label=None, offset=None, timestamp_event=None, meta_event=None, start=None, elements=None):
        """Add address with optional parameters to the properties for synchronous readout.

        Args:
            address (str): The DOOCS address to add to the properties for synchronous readout.
            label (str, optional): Optional label for DOOCS address as a means for alternative property access.
            offset (int, optional): Optional offset to be applied on event numbers for synchronization.
            timestamp_event (bool, optional): Optional state to determine using the timestamp as an alternative event.
            meta_event (str, optional): Optional Meta property to replace the default event with.
            start (int, optional): Optional start position for reading out array-like types.
            elements (int, optional): Optional number of elements to read out for array-like types.

        Returns:
            None

        Raises:
            DoocspieException: Doocspie related exception for address duplication.
        """
        self._synchronizer.add(address, label, offset, timestamp_event, meta_event, start, elements)
        self._synchronizer_to_update = True

    def add_axis_labels(self, source, x_label=None, y_label=None):
        """Add axis labels to the given data source (either a property added or a registered function).

        Args:
            source (str): The data source (property or registered function) to add the axis labels to.
            x_label (str, optional): Optional x-label for the given source.
            y_label (str, optional): Optional y-label for the given source.

        Returns:
            None
        """
        source_label = self._synchronizer.get_label_of(source)
        if source_label:
            self._axis_labels[source_label] = x_label, y_label
        else:
            self._axis_labels[source] = x_label, y_label

    def update(self):
        """Update the newly added properties to for synchronous readout.

        Returns:
            None
        """
        self._synchronizer.update()

    def register(self, arg):
        """Decorator with optional label supplied via argument for registration of a function.

        Args:
            arg (str or function): Either a label (str) attached to the function or the function itself.

        Returns:
            None

        Raises:
            DoocspieException: Doocspie related exception for an already registered function label.
        """
        if callable(arg):
            self._register(arg.__name__, arg)
            return

        def wrapper(func):
            self._register(arg, func)

        return wrapper

    def _register(self, label, function):
        """Helper method to register a function with its label."""
        if label not in self._registered_functions:
            self._registered_functions[label] = function
        else:
            self._raise_doocspie_exception(label)

    @staticmethod
    def _raise_doocspie_exception(label):
        """Helper method to raise a doocspie exception for an already registered function label."""
        raise DoocspieException("registered function label '" + label + "' already exists", label)

    def __str__(self):
        """Special method to return a properly formatted string representation of the datalyzer."""
        return ("Datalyzer(" +
                "actual_properties=" + str(self.actual_properties) + ", " +
                "registered_functions=" + str(self.registered_functions) + ")")

    def startup(self):
        """Start up the datalyzer GUI application.

        Returns:
            None
        """
        if self._synchronizer_to_update:
            self._synchronizer.update()
        self._check_properties_added()
        self._check_unique_labels()
        self._check_axis_labels()
        self._startup_application()

    def _check_properties_added(self):
        """Helper method to check if properties were added."""
        if not self._synchronizer.actual_properties:
            raise DoocspieException("no properties added")

    def _check_unique_labels(self):
        """Helper method to check unique labels."""
        for label in self._registered_functions:
            if label in self._synchronizer.actual_properties:
                self._raise_doocspie_exception(label)

    def _check_axis_labels(self):
        """Helper method to check if axis labels can be added to data sources."""
        for source in self._axis_labels:
            if source not in self._synchronizer.actual_properties + tuple(self._registered_functions.keys()):
                raise DoocspieException("cannot add axis labels for data source '" + source + "'")

    def _startup_application(self):
        """Helper method to start up the actual GUI application."""
        signal.signal(signal.SIGINT, signal.SIG_DFL)  # handles Ctrl-C interrupts via core dump and exit

        resources_path = os.path.dirname(os.path.abspath(__file__)) + "/resources/"

        application = QApplication([__application_name__])
        application.setWindowIcon(QIcon(resources_path + "datalyzer.png"))

        image_data = ImageData()
        vector_data = VectorData()
        dual_vector_data = DualVectorData()
        scalar_data = ScalarData()

        io_service = IOService(self._synchronizer, self._registered_functions,
                               image_data, vector_data, dual_vector_data, scalar_data)

        datalyzer_tool = DatalyzerTool(__application_name__, __export_directory__, __version__, __elogs__,
                                       io_service, self._axis_labels)
        datalyzer_tool.show_window()

        sys.exit(application.exec())
