"""Module with readout class for higher level output from DOOCS.

This module provides the readout class for higher level output from the DOOCS control system.
"""
import numpy as np

from doocspie.doocspie.meta import Meta


class Readout:
    """Readout class for higher level output from DOOCS.

    This class provides the readout for higher level output from the DOOCS control system.
    """

    def __init__(self, pydoocs_output, timestamp_event, miscellaneous_event):
        """Constructor of the readout class.

        This constructor initializes the instance using the pydoocs output and a miscellaneous event.

        Args:
            pydoocs_output (dict): The output data provided by pydoocs.
            timestamp_event (bool): The state to determine using the timestamp as an alternative event.
            miscellaneous_event (str): The miscellaneous event to be used as an alternative event.
        """
        self._data = self._get_data(pydoocs_output)
        self._type = pydoocs_output["type"]
        self._timestamp = pydoocs_output["timestamp"]
        self._event = self._get_event_data(pydoocs_output, timestamp_event, miscellaneous_event)
        self._meta = Meta(pydoocs_output["miscellaneous"])

    @staticmethod
    def _get_data(pydoocs_output):
        """Helper method to return the data, and in case of spectral types omit the time axis."""
        if pydoocs_output["type"] in ("SPECTRUM", "GSPECTRUM"):
            return pydoocs_output["data"][:, 1]
        return pydoocs_output["data"]

    @staticmethod
    def _get_event_data(pydoocs_output, timestamp_event, miscellaneous_event):
        """Helper method to return the specific event number."""
        if timestamp_event:
            return round(pydoocs_output["timestamp"], 1)  # events are generated with 10 Hz, i.e. every 0.1 sec
        if miscellaneous_event:
            return pydoocs_output["miscellaneous"][miscellaneous_event]
        return pydoocs_output["macropulse"]

    @property
    def data(self):
        """property dependent: The returned DOOCS data."""
        return self._data

    @property
    def type(self):
        """str: The DOOCS data type of the returned data."""
        return self._type

    @property
    def timestamp(self):
        """float: The timestamp linked to the returned data."""
        return self._timestamp

    @property
    def event(self):
        """int: The event number linked to the returned data."""
        return self._event

    @property
    def meta(self):
        """Meta: The metadata of the readout."""
        return self._meta

    def __str__(self):
        """Special method to return a properly formatted string representation of the readout."""
        return ("Readout(" +
                "data=" + str(self._data) + ", " +
                "type='" + self._type + "', " +
                "timestamp=" + str(self._timestamp) + ", " +
                "event=" + str(self._event) + ", " +
                "meta=" + str(self._meta) + ")")

    def __len__(self):
        """Special method to return the length of the data (in case of array-like types)."""
        return len(self._data)

    def __eq__(self, other):
        """Special method to check for equality by using the '==' operator."""
        if isinstance(other, Readout):
            return (np.all(self._data == other.data) and self._type == other.type and
                    self._timestamp == other.timestamp and self._event == other.event and self._meta == other.meta)
        return False
