"""Module with functions for higher level I/O within DOOCS.

This module provides the functions for higher level I/O operations with the DOOCS control system.
"""
from datetime import datetime

from doocspie import core
from doocspie.doocspie.doocspie_exception import DoocspieException
from doocspie.doocspie.history import History
from doocspie.doocspie.readout import Readout


def ls(address):
    """List the members of the given 4-layer address scheme 'facility/device/location/property'.

    Args:
        address (str): The address part, optionally including the * wildcard, to query the members from.

    Returns:
        tuple: The queried members of the 4-layer DOOCS address scheme.

    Raises:
        DoocspieException: DOOCS related exception for non-exiting address parts to query from.
    """
    original_address = address
    if not address or address[-1] == "/":
        address += "*"

    try:
        names = core.names(address)
    except core.DoocsException as exc:
        raise DoocspieException(exc, original_address) from None

    prefix = address[:address[:address.find("*")].rfind("/") + 1]  # address prefix taking into account first asterisk
    return tuple(prefix + name.split()[0] for name in names)


def get(address, event=None, ignore_event_mismatch=False, timestamp_event=False, meta_event=None, start=None,
        elements=None):
    """Get the readout for a given DOOCS address with optional parameters (see 'Args' below).

    Args:
        address (str): The property address of the server to get the readout from.
        event (int, optional): Optional event to get the data from.
        ignore_event_mismatch (bool, optional): Optional state for ignoring event mismatches.
        timestamp_event (bool, optional): Optional state to determine using the timestamp as an alternative event.
        meta_event (str, optional): Optional Meta property to replace the default event with.
        start (int, optional): Optional start position for reading out array-like types.
        elements (int, optional): Optional number of elements to read out for array-like types.

    Returns:
        Readout: An instance of the readout object with its properties 'data', 'type', 'timestamp', 'event' and 'meta'.

    Raises:
        DoocspieException: Context-dependent exception related to DOOCS, pydoocs or doocspie itself.
    """
    try:
        pydoocs_output = _get_pydoocs_output(address, event, ignore_event_mismatch, start, elements)
    except (core.DoocsException, core.PyDoocsException) as exc:
        raise DoocspieException(exc, address) from None

    _check_valid_event_options(timestamp_event, meta_event, address)
    if meta_event and meta_event not in pydoocs_output["miscellaneous"]:
        raise DoocspieException("'meta_event' must be in " + str(tuple(pydoocs_output["miscellaneous"].keys())),
                                address)
    return Readout(pydoocs_output, timestamp_event, meta_event)


def _get_pydoocs_output(address, event, ignore_event_mismatch, start, elements):
    """Helper function to return pydoocs output for the given parameters."""
    if start or elements:
        if event is None:
            event = -1
        if start is None:
            start = -1
        if elements is None:
            elements = -1
        pydoocs_output = core.read(address, parameters=[event, start, -1, elements],  # 'increment' always default (-1)
                                   ignore_macropulse_mismatch=ignore_event_mismatch)
        if not (pydoocs_output["type"] in ("SPECTRUM", "GSPECTRUM") or pydoocs_output["type"].startswith("A_")):
            raise DoocspieException("'start' and/or 'elements' only have effect on 'SPECTRUM', 'GSPECTRUM' or 'A_*'",
                                    address)
    elif event:
        pydoocs_output = core.read(address, macropulse=event, ignore_macropulse_mismatch=ignore_event_mismatch)
    else:
        pydoocs_output = core.read(address)
    return pydoocs_output


def _check_valid_event_options(timestamp_event, meta_event, address):
    """Helper method to check for valid event options."""
    if timestamp_event and meta_event:
        raise DoocspieException("'meta_event' has no effect when 'timestamp_event' is being used", address)


def set(address, value, allow_resizing=False):
    """Set the content of DOOCS server properties with the given value and with optional resizing.

    Args:
        address (str): The property address of the server to set the value to.
        value (property dependent): The value to set the server property to.
        allow_resizing(bool, optional): The optional state for allowing resizing of array-like DOOCS types.

    Returns:
        None

    Raises:
        DoocspieException: DOOCS or pydoocs related exception for server errors or wrong input.
    """
    try:
        if allow_resizing:
            core.write(address, value, allow_resizing=allow_resizing)
        else:
            core.write(address, value)
    except (core.DoocsException, core.PyDoocsException) as exc:
        raise DoocspieException(exc, address) from None


def get_history(address, begin=None, end=None):
    """Get the history of a given DOOCS history address with optional 'begin' and 'end' datetimes.

    Args:
        address (str): The history property address of the server to get the history readout from.
        begin (datetime, optional): The optional beginning datetime for the history to get data from.
        end (datetime, optional): The optional ending datetime for the history to get data from.

    Returns:
        History: An instance of the history object with properties 'times', 'values' and 'states'.

    Raises:
        ValueError: Exception for wrong DOOCS history address.
        TypeError: Exception for wrong 'begin' and 'end' types.
        DoocspieException: Context-dependent exception related to DOOCS, pydoocs or doocspie itself.
    """
    if not address.endswith(".HIST"):
        raise ValueError("address must have '.HIST' extension")

    if begin is not None and not isinstance(begin, datetime):
        raise TypeError("'begin' must be of type 'datetime'")
    if end is not None and not isinstance(end, datetime):
        raise TypeError("'end' must be of type 'datetime'")
    if (begin is not None and end is None) or (begin is None and end is not None):
        raise DoocspieException("'begin' and 'end' must be given simultaneously or not at all", address)

    try:
        if begin is not None and end is not None:
            if begin >= end:
                raise DoocspieException("'begin' and 'end' must be in chronological order", address)
            return History(core.read(address, parameters=[datetime.timestamp(begin), datetime.timestamp(end),
                                                          256, 0]))  # the 'magic' numbers 256 and 0 are DOOCS specific
        return History(core.read(address))
    except core.DoocsException as exc:
        raise DoocspieException(exc, address) from None
