"""Module with history class for higher level history output from DOOCS.

This module provides the history class for higher level history output from DOOCS.
"""
from datetime import datetime

import numpy as np


class History:
    """History class for higher level history output from DOOCS.

    This class provides the higher level history output from the DOOCS control system.
    """

    def __init__(self, pydoocs_output):
        """Constructor of the history class.

        This constructor initializes the instance using the pydoocs output for DOOCS histories.

        Args:
            pydoocs_output (dict): The output data for DOOCS history properties provided by pydoocs.
        """
        self._times = np.array([datetime.fromtimestamp(element[0]) for element in pydoocs_output["data"]])
        self._values = np.array([element[1] for element in pydoocs_output["data"]])
        self._states = np.array([element[2] for element in pydoocs_output["data"]], dtype=int)

    @property
    def times(self):
        """numpy.ndarray: The datetimes of the returned history."""
        return self._times

    @property
    def values(self):
        """numpy.ndarray: The data values of the returned history."""
        return self._values

    @property
    def states(self):
        """numpy.ndarray: The state information of the returned history."""
        return self._states

    def __str__(self):
        """Special method to return a properly formatted string representation of the history."""
        return ("History(" +
                "times=" + str(np.array([str(time) for time in self._times])) + ", " +
                "values=" + str(self._values) + ", " +
                "states=" + str(self._states) + ")")

    def __len__(self):
        """Special method to return the length of the history."""
        return len(self._times)
