"""Module with exception class for higher level I/O within DOOCS.

This module provides the custom exception class for higher level I/O operations with DOOCS.
"""
from doocspie import core


class DoocspieException(Exception):
    """Exception class for higher level I/O within DOOCS.

    This class provides the custom exception for higher level I/O operations with DOOCS.
    """

    def __init__(self, exception, address=None):
        """Constructor of the custom exception class.

        This constructor initializes the instance using the causing low-level exception and an optional
        address.

        Args:
            exception (any Exception): The causing low-level exception.
            address (str, optional): The exception causing address.
        """
        if isinstance(exception, core.DoocsException):
            self._exception = type(exception).__name__ + " with " + str(exception)
            self._code = exception.args[0]["code"]
            self._message = exception.args[0]["message"]
        elif isinstance(exception, core.PyDoocsException):
            exception_message = self._get_exception_message(exception)
            self._exception = type(exception).__name__ + " with '" + exception_message + "'"
            self._code = None
            self._message = exception_message
        else:
            self._exception = exception
            self._code = None
            self._message = self._exception
        self._address = address
        super().__init__(self._exception)

    @staticmethod
    def _get_exception_message(exception):
        """Helper method for transforming exception messages with 'macropulse' being replaced by 'event'."""
        mismatch = exception.args[0].split()[-1]
        if exception.args[0] == "macropulse mismatch: (returned - requested) = " + mismatch:
            return f"event mismatch: (returned - requested) = {mismatch}"
        if exception.args[0] == "wrong macropulse type":
            return "wrong event type"
        if exception.args[0] == "wrong ignore_macropulse_mismatch type":
            return "wrong ignore_event_mismatch type"
        return str(exception)

    @property
    def code(self):
        """int: The error code of the exception. This is only relevant for DOOCS related exceptions."""
        return self._code

    @property
    def message(self):
        """str: The message of the exception."""
        return self._message

    @property
    def address(self):
        """str: The exception causing address."""
        return self._address
