"""Module with daq monitor class for simplified readout using DAQ monitors.

This module provides the daq monitor class for simplified readout using the DAQ monitors.
"""
import warnings

import doocspie
from doocspie.doocspie.doocspie_exception import DoocspieException


class DaqMonitor:
    """DaqMonitor class for simplified readout using DAQ monitors.

    This class provides the daq monitor for simplified readout using DAQ monitors.
    """
    warnings.formatwarning = lambda msg, *args, **kwargs: f"Warning: {msg}\n"

    def __init__(self, device, facility="FLASH.DAQ"):
        """Constructor of the daq monitor class.

        This constructor initializes the instance with the device and optional DAQ monitor facility.

        Args:
            device (str): The device address part of the DAQ monitor.
            facility (str, optional): The optional facility of the DAQ monitor.
        """
        self._device = device
        self._facility = facility
        self._state_address = self._facility + "/" + self._device + "/DAQFSM/DAQ.FSM_STATE_A"
        if not self.is_running():
            warnings.warn(f"DAQ Monitor '{self._facility}/{self._device}' is not running")
        self._address_mapping = self._get_address_mapping()

    def _get_address_mapping(self):
        """Helper method to provide the address mapping between DOOCS property and DAQ monitor address."""
        property_suffix = ".DAQ.SOURCE"
        try:
            addresses = doocspie.ls(self._facility + "/" + self._device + "/MONITOR/*" + property_suffix)
            return {doocspie.get(address).data: address[:-len(property_suffix)] for address in addresses
                    if doocspie.get(address).data}
        except DoocspieException as exc:
            raise DoocspieException("cannot get address mapping for given DAQ monitor", self._state_address) from exc

    @property
    def properties(self):
        """tuple: The properties available via the DAQ monitor."""
        return tuple(self._address_mapping.keys())

    @classmethod
    def available_devices(cls, facility="FLASH.DAQ"):
        """List the available devices (DAQ monitors) for a given optional facility.

        Returns:
            tuple: The available devices (DAQ monitors) for a given optional facility.

        Raises:
            DoocspieException: Doocspie related exception for non-listable DAQ monitors.
        """
        try:
            return tuple(monitor.split("/")[-1] for monitor in doocspie.ls(facility + "/*.MONITOR"))
        except DoocspieException as exc:
            raise DoocspieException("cannot list available devices for given facility", facility) from exc

    def is_running(self):
        """Return the DAQ monitor's 'running' state.

        Returns:
            bool: The 'running' state of the DAQ monitor.

        Raises:
            DoocspieException: Doocspie related exception for inaccessible 'running' state.
        """
        try:
            return doocspie.get(self._state_address).data == "RUN"
        except DoocspieException as exc:
            raise DoocspieException("cannot get 'running' state of given DAQ monitor", self._state_address) from exc

    def refresh(self):
        """Refresh the DAQ monitor in order to detect changes of the address mapping.

        Returns:
            None
        """
        if not self.is_running():
            warnings.warn(f"DAQ Monitor '{self._facility}/{self._device}' is not running")
        self._address_mapping = self._get_address_mapping()

    def map(self, address):
        """Map the DOOCS property address to the address of the associated DAQ monitor.

        Args:
            address (str): The property address to get the mapped DAQ monitor address for.

        Returns:
            str: The address of the associated DAQ monitor mapped to the given DOOCS property address.

        Raises:
            DoocspieException: Doocspie related exception for non-existing addresses to be mapped.
        """
        if address not in self._address_mapping:
            raise DoocspieException("cannot find mapping for given address", address)
        return self._address_mapping[address]

    def get(self, address, event=None, ignore_event_mismatch=False, timestamp_event=False, meta_event=None, start=None,
            elements=None):
        """Get the readout for a given DOOCS address with optional parameters (see 'Args' below).

        Args:
            address (str): The property address of the server to get the readout from.
            event (int, optional): Optional event to get the data from.
            ignore_event_mismatch (bool, optional): Optional state for ignoring event mismatches.
            timestamp_event (bool, optional): Optional state to determine using the timestamp as an alternative event.
            meta_event (str, optional): Optional Meta property to replace the default event with.
            start (int, optional): Optional start position for reading out array-like types.
            elements (int, optional): Optional number of elements to read out for array-like types.

        Returns:
            Readout: An instance of the readout object with its properties 'data', 'type', 'timestamp', 'event' and
            'meta'.

        Raises:
            DoocspieException: Context-dependent exception related to DOOCS, pydoocs or doocspie itself.
        """
        try:
            return doocspie.get(self.map(address), event, ignore_event_mismatch, timestamp_event, meta_event, start,
                                elements)
        except DoocspieException as exc:
            raise DoocspieException("cannot get readout for given address", address) from exc

    def __str__(self):
        """Special method to return a properly formatted string representation of the daq monitor."""
        return ("DaqMonitor(" +
                "facility=" + str(self._facility) + ", " +
                "device=" + str(self._device) + ", " +
                "properties=" + str(self.properties) + ", " +
                "is_running=" + str(self.is_running()) + ")")
