"""Module with train event class for higher level synchronous readout from DOOCS.

This module provides the train event class for higher level synchronous readout from DOOCS.
"""
from doocspie.doocspie.doocspie_exception import DoocspieException


class TrainEvent:
    """TrainEvent class for higher level synchronous readout from DOOCS.

    This class provides the train event for higher level synchronous readout from DOOCS.
    """

    def __init__(self, readouts, properties):
        """Constructor of the train event class.

        This constructor initializes the instance using the readouts and its related properties.

        Args:
            readouts (dict): The readouts for the related properties.
            properties (tuple): The properties related to the readouts.
        """
        self._readouts = readouts
        self._properties = properties
        if self._properties:
            self._id = self._readouts[self._properties[0]].event  # all properties have the same ID by definition
        else:
            self._id = None

    @property
    def properties(self):
        """tuple: The properties that can be returned via 'get'."""
        return self._properties

    @property
    def id(self):
        """int: The id of the current train event."""
        return self._id

    def get(self, property):
        """Get the readout from the train event for the given property.

        Args:
            property (str): The property for the readout to get from the train event.

        Returns:
            Readout: An instance of the readout object for the requested property.

        Raises:
            DoocspieException: Doocspie related exception for non-existing property requested.
        """
        try:
            return self._readouts[property]
        except KeyError as err:
            raise DoocspieException("readout for " + str(err) + " does not exist", property) from None

    def __str__(self):
        """Special method to return a properly formatted string representation of the train event."""
        return "TrainEvent(" + "id=" + str(self._id) + ", " + "properties=" + str(self._properties) + ")"
