"""Module with train abo class for higher level synchronous readout from DOOCS.

This module provides the train abo class for higher level synchronous readout from DOOCS.
"""
from doocspie.abo.synchronizer import Synchronizer
from doocspie.abo.train_event import TrainEvent


class TrainAbo:
    """TrainAbo class for higher level synchronous readout from DOOCS.

    This class provides the train abo for higher level synchronous readout from DOOCS.
    """

    def __init__(self, properties=None, train_events=None, timeout_seconds=10, buffer_size=32, allow_zero_events=False):
        """Constructor of the train abo class.

        This constructor initializes the instance with optional parameters (see 'Args' below).

        Args:
            properties (tuple or dict, optional): The optional properties to get train events with readouts for.
            train_events (int, optional): The optional number of train events to synchronously readout.
            timeout_seconds (int): The optional timeout seconds for unsuccessful readout.
            buffer_size (int): The optional number of buffers to be used for storing previous readouts.
            allow_zero_events (bool, optional): The optional state for allow zero train events.
        """
        self._train_events = train_events
        self._train_event = 0
        self._synchronizer = Synchronizer(properties, timeout_seconds, buffer_size, allow_zero_events)
        self._stop_loop = None

    @property
    def train_events(self):
        """int: The number of total train events in the train abo."""
        return self._train_events

    @property
    def train_event(self):
        """int: The number of the current train event in the train abo."""
        return self._train_event

    @property
    def usable_properties(self):
        """tuple: The properties that can be used for synchronized readout in the train abo."""
        return self._synchronizer.usable_properties

    @property
    def unusable_properties(self):
        """tuple: The properties that cannot be used for synchronized readout in the train abo."""
        return self._synchronizer.unusable_properties

    @property
    def zero_event_properties(self):
        """tuple: The properties that have an event number of 'zero'."""
        return self._synchronizer.zero_event_properties

    @property
    def actual_properties(self):
        """tuple: The properties that can actually be readout and returned via TrainEvent's 'get'."""
        return self._synchronizer.actual_properties

    def add(self, address, label=None, offset=None, timestamp_event=None, meta_event=None, start=None, elements=None):
        """Add address with optional parameters to the properties for synchronous readout.

        Args:
            address (str): The DOOCS address to add to the properties for synchronous readout.
            label (str, optional): Optional label for DOOCS address as a means for alternative property access.
            offset (int, optional): Optional offset to be applied on event numbers for synchronization.
            timestamp_event (bool, optional): Optional state to determine using the timestamp as an alternative event.
            meta_event (str, optional): Optional Meta property to replace the default event with.
            start (int, optional): Optional start position for reading out array-like types.
            elements (int, optional): Optional number of elements to read out for array-like types.

        Returns:
            None

        Raises:
            DoocspieException: Doocspie related exception for address duplication.
        """
        self._synchronizer.add(address, label, offset, timestamp_event, meta_event, start, elements)

    def update(self):
        """Update the newly added properties for synchronous readout.

        Returns:
            None
        """
        self._synchronizer.update()

    def reset(self):
        """Reset the train abo in order to use it again.

        Returns:
            None
        """
        self._train_event = 0

    def start_loop(self, function):
        """Start the loop the train abo and execute the given function once there is a new train event.

        Args:
            function (function): The function to execute for a new train event.

        Returns:
            None
        """
        self._check_stop_iteration()
        self._stop_loop = False
        while not self._all_train_events_generated() and not self._stop_loop:
            self._train_event += 1
            function(TrainEvent(self._synchronizer.get(), self._synchronizer.actual_properties))

    def stop_loop(self):
        """Stop the loop of the train abo and its execution of the supplied function.

        Returns:
            None
        """
        self._stop_loop = True

    def __next__(self):
        """Special method to provide the next train event for the train abo iterator."""
        self._check_stop_iteration()
        self._train_event += 1
        return TrainEvent(self._synchronizer.get(), self._synchronizer.actual_properties)

    def _check_stop_iteration(self):
        """Helper method to check and stop the train event iterator."""
        if self._all_train_events_generated():
            raise StopIteration("all train events have been generated")

    def _all_train_events_generated(self):
        """Helper method to query if all train events haven been generated."""
        return self._train_events is not None and self._train_event >= self._train_events

    def __iter__(self):
        """Special method to provide a train abo iterator."""
        return self

    def __str__(self):
        """Special method to return a properly formatted string representation of the train abo."""
        return ("TrainAbo(" +
                "train_events=" + str(self._train_events) + ", " +
                "train_event=" + str(self._train_event) + ", " +
                "actual_properties=" + str(self.actual_properties) + ")")
