"""Module with synchronizer class for higher level synchronous readout from DOOCS.

This module provides the synchronizer class for higher level synchronous readout from DOOCS.
"""
import time
from collections import defaultdict, namedtuple

import numpy as np

import doocspie


class Synchronizer:
    """Synchronizer class for higher level synchronous readout from DOOCS.

    This class provides the synchronizer for higher level synchronous readout from DOOCS.
    """

    _LABEL_KEY = "label"
    _OFFSET_KEY = "offset"
    _TIMESTAMP_EVENT_KEY = "timestamp_event"
    _META_EVENT_KEY = "meta_event"
    _START_KEY = "start"
    _ELEMENTS_KEY = "elements"
    _OPTIONS = (_LABEL_KEY, _OFFSET_KEY, _TIMESTAMP_EVENT_KEY, _META_EVENT_KEY, _START_KEY, _ELEMENTS_KEY)

    _Event = namedtuple("Event", ("number", "exception"))
    _Usable = namedtuple("Usable", ("address", _LABEL_KEY, _OFFSET_KEY, _TIMESTAMP_EVENT_KEY, _META_EVENT_KEY,
                                    _START_KEY, _ELEMENTS_KEY))
    _Unusable = namedtuple("Unusable", ("address", _LABEL_KEY, "reason"))

    _MINIMUM_EVENTS = 5

    def __init__(self, properties=None, timeout_seconds=10, buffer_size=32, allow_zero_events=False):
        """Constructor of the synchronizer class.

        This constructor initializes the instance with optional parameters (see 'Args' below).

        Args:
            properties (tuple or dict, optional): The optional properties to get train events with readouts for.
            timeout_seconds (int): The optional timeout seconds for unsuccessful readout.
            buffer_size (int): The optional number of buffers to be used for storing previous readouts.
            allow_zero_events (bool, optional): The optional state for allow zero train events.
        """
        self._properties = self._conditionally_make_dict(properties)
        self._check_available_options()
        self._timeout_seconds = timeout_seconds
        self._buffer_size = buffer_size
        self._allow_zero_events = allow_zero_events
        self._usable_properties, self._unusable_properties, self._zero_event_properties = (), (), ()
        self._actual_properties = ()
        self._buffers = ()
        self._properties_to_update = None
        self._last_event_returned = None
        self.update()
        self._check_valid_options()

    @staticmethod
    def _conditionally_make_dict(properties):
        """Helper method to make a dict out of the properties if it is not already a dict."""
        if properties is None:
            return {}
        if isinstance(properties, dict):
            return properties.copy()
        if isinstance(properties, (tuple, list)):
            return dict.fromkeys(properties, {})
        raise doocspie.DoocspieException("properties must be of type 'dict', 'tuple' or 'list'")

    def _check_available_options(self):
        """Helper method to check for available options."""
        for address, options in self._properties.items():
            if isinstance(options, dict):
                self._check_valid_event_options(options.get(self._TIMESTAMP_EVENT_KEY),
                                                options.get(self._META_EVENT_KEY), address)
                for option in options.keys():
                    if option not in self._OPTIONS:
                        raise doocspie.DoocspieException("option '" + option + "' is not available", address)

    @property
    def timeout_seconds(self):
        """int: The timeout seconds for unsuccessful readout."""
        return self._timeout_seconds

    @property
    def usable_properties(self):
        """tuple: The properties that can be used for synchronized readout in the train abo."""
        return self._usable_properties

    @property
    def unusable_properties(self):
        """tuple: The properties that cannot be used for synchronized readout in the train abo."""
        return self._unusable_properties

    @property
    def zero_event_properties(self):
        """tuple: The properties that have an event number of 'zero'."""
        return self._zero_event_properties

    @property
    def actual_properties(self):
        """tuple: The properties that can actually be readout."""
        return self._actual_properties

    def update(self):
        """Update the newly added properties for synchronous readout.

        Returns:
            None
        """
        self._usable_properties, self._unusable_properties, self._zero_event_properties = self._split_properties()
        self._actual_properties = self._get_actual_properties()
        self._buffers = tuple({} for _ in range(len(self._usable_properties)))
        self._properties_to_update = False
        self._last_event_returned = None

    def _split_properties(self):
        """Helper method to split the properties in particular categories."""
        usable_properties, unusable_properties, zero_event_properties = [], [], []
        labels = []
        for address, events in self._get_address_events(self._properties):
            label = self._get_unique_label(address, labels)
            if len(set(events)) == 1:
                event = events[0]
                if event.exception:
                    unusable_properties.append(self._Unusable(address, label, event.exception))
                elif event.number:
                    unusable_properties.append(self._Unusable(address, label, "static event number"))
                else:
                    unusable_properties.append(self._Unusable(address, label, "zero event number"))
                    if self._allow_zero_events:
                        zero_event_properties.append(
                            self._Usable(address, label, None, self._get_timestamp_event(address),
                                         self._get_meta_event(address), self._get_start(address),
                                         self._get_elements(address)))
            else:
                usable_properties.append(
                    self._Usable(address, label, self._get_offset(address), self._get_timestamp_event(address),
                                 self._get_meta_event(address), self._get_start(address), self._get_elements(address)))
        return tuple(usable_properties), tuple(unusable_properties), tuple(zero_event_properties)

    def _get_address_events(self, properties):
        """Helper method to get address events."""
        address_events = defaultdict(list)
        event_numbers_changed_sufficiently = False
        start = time.time()
        while properties:
            for address in properties:
                try:
                    event = self._Event(doocspie.get(address, timestamp_event=self._get_timestamp_event(address),
                                                     meta_event=self._get_meta_event(address),
                                                     start=self._get_start(address),
                                                     elements=self._get_elements(address)).event, None)
                except doocspie.DoocspieException as exc:
                    event = self._Event(np.nan, exc.message)
                address_events[address].append(event)

                if (address_events[address][-1].number - address_events[address][0].number) >= self._MINIMUM_EVENTS:
                    event_numbers_changed_sufficiently = True
            if event_numbers_changed_sufficiently:
                break
            if (time.time() - start) > self._timeout_seconds:
                raise doocspie.DoocspieException("no usable properties found")
        return address_events.items()

    def _get_unique_label(self, address, labels):
        """Helper method to get optional and unique label."""
        options = self._properties[address]
        if isinstance(options, dict) and self._LABEL_KEY in options:
            label = options[self._LABEL_KEY]
            if label and label in labels:
                raise doocspie.DoocspieException(self._LABEL_KEY + " '" + label + "' already exists", address)
            labels.append(label)
        else:
            label = None
        return label

    def _get_offset(self, address):
        """Helper method to get optional offset."""
        options = self._properties[address]
        if isinstance(options, dict) and self._OFFSET_KEY in options:
            return options[self._OFFSET_KEY]
        return None

    def _get_timestamp_event(self, address):
        """Helper method to get optional timestamp event."""
        options = self._properties[address]
        if isinstance(options, dict) and self._TIMESTAMP_EVENT_KEY in options:
            return options[self._TIMESTAMP_EVENT_KEY]
        return None

    def _get_meta_event(self, address):
        """Helper method to get optional meta event."""
        options = self._properties[address]
        if isinstance(options, dict) and self._META_EVENT_KEY in options:
            return options[self._META_EVENT_KEY]
        return None

    def _get_start(self, address):
        """Helper method to get optional start."""
        options = self._properties[address]
        if isinstance(options, dict) and self._START_KEY in options:
            return options[self._START_KEY]
        return None

    def _get_elements(self, address):
        """Helper method to get optional elements."""
        options = self._properties[address]
        if isinstance(options, dict) and self._ELEMENTS_KEY in options:
            return options[self._ELEMENTS_KEY]
        return None

    def _get_actual_properties(self):
        """Helper method to get actual properties."""
        actual_properties = []
        for property in self._usable_properties + self._zero_event_properties:
            actual_properties.append(property.address)
            if property.label:
                actual_properties.append(property.label)
        return tuple(actual_properties)

    def add(self, address, label=None, offset=None, timestamp_event=None, meta_event=None, start=None, elements=None):
        """Add address with optional parameters to the properties for synchronous readout.

        Args:
            address (str): The DOOCS address to add to the properties for synchronous readout.
            label (str, optional): Optional label for DOOCS address as a means for alternative property access.
            offset (int, optional): Optional offset to be applied on event numbers for synchronization.
            timestamp_event (bool, optional): Optional state to determine using the timestamp as an alternative event.
            meta_event (str, optional): Optional Meta property to replace the default event with.
            start (int, optional): Optional start position for reading out array-like types.
            elements (int, optional): Optional number of elements to read out for array-like types.

        Returns:
            None

        Raises:
            DoocspieException: Doocspie related exception for address duplication.
        """
        if address not in self._properties:
            self._check_valid_event_options(timestamp_event, meta_event, address)
            options = {}
            if label:
                options[self._LABEL_KEY] = label
            if offset:
                options[self._OFFSET_KEY] = offset
            if timestamp_event:
                options[self._TIMESTAMP_EVENT_KEY] = timestamp_event
            if meta_event:
                options[self._META_EVENT_KEY] = meta_event
            if start:
                options[self._START_KEY] = start
            if elements:
                options[self._ELEMENTS_KEY] = elements
            self._properties[address] = options
            self._properties_to_update = True
        else:
            raise doocspie.DoocspieException("address '" + address + "' already exists", address)

    @staticmethod
    def _check_valid_event_options(timestamp_event, meta_event, address):
        """Helper method to check for valid event options."""
        if timestamp_event and meta_event:
            raise doocspie.DoocspieException("'meta_event' has no effect when 'timestamp_event' is being used", address)

    def get(self):
        """Get the synchronous readouts mapped by its address or label for the particular properties.

        Returns:
            dict: The address (key) and readout (value) for the particular properties.

        Raises:
            DoocspieException: Doocspie related exception for synchronization timeout or no usable properties.
        """
        if self._properties_to_update:
            self.update()

        start = time.time()
        while self._usable_properties:
            event_matched_readouts = self.get_event_matched_readouts()
            if event_matched_readouts:
                return event_matched_readouts
            if (time.time() - start) > self._timeout_seconds:
                raise doocspie.DoocspieException("synchronization timeout")
        raise doocspie.DoocspieException("no usable properties found")

    def get_event_matched_readouts(self):
        """Get the event matched readouts mapped by its address or label for the particular properties.

        Returns:
            dict: The address (key) and readout (value) for the particular properties.
        """
        self._fill_buffers()
        event_matched_readouts = self._find_event_matched_readouts()
        if event_matched_readouts:
            if self._allow_zero_events:
                self._add_zero_event_readouts(event_matched_readouts)
        return event_matched_readouts

    def _fill_buffers(self):
        """Helper method to fill the buffers."""
        for usable_property, buffer in zip(self._usable_properties, self._buffers):
            try:
                readout = doocspie.get(usable_property.address, timestamp_event=usable_property.timestamp_event,
                                       meta_event=usable_property.meta_event, start=usable_property.start,
                                       elements=usable_property.elements)
                event = readout.event
                if usable_property.offset:
                    event += usable_property.offset
                if event not in buffer:
                    if len(buffer) == self._buffer_size:
                        del buffer[list(buffer)[0]]
                    buffer[event] = readout
            except doocspie.DoocspieException:
                continue

    def _find_event_matched_readouts(self):
        """Helper method to find readouts matched based on events."""
        event_matched_readouts = {}
        for event in self._buffers[0]:
            events_in_buffer = [event in buffer for buffer in self._buffers]
            if all(events_in_buffer) and (self._last_event_returned is None or event > self._last_event_returned):
                self._last_event_returned = event
                for index, buffer in enumerate(self._buffers):
                    event_matched_readouts[self._usable_properties[index].address] = buffer[event]
                    if self._usable_properties[index].label:
                        event_matched_readouts[self._usable_properties[index].label] = buffer[event]
                    del buffer[event]
                return event_matched_readouts
        return event_matched_readouts

    def _add_zero_event_readouts(self, readouts):
        """Helper method to add zero event readouts."""
        for zero_event_property in self._zero_event_properties:
            readouts[zero_event_property.address] = doocspie.get(zero_event_property.address,
                                                                 timestamp_event=zero_event_property.timestamp_event,
                                                                 meta_event=zero_event_property.meta_event,
                                                                 start=zero_event_property.start,
                                                                 elements=zero_event_property.elements)
            if zero_event_property.label:
                readouts[zero_event_property.label] = readouts[zero_event_property.address]

    def _check_valid_options(self):
        """Helper method to check for valid options."""
        for property in self._usable_properties + self._zero_event_properties:
            # raise an exception if the DOOCS property (address) does not have optional parameters
            doocspie.get(property.address, timestamp_event=property.timestamp_event, meta_event=property.meta_event,
                         start=property.start, elements=property.elements)

    def get_label_of(self, source):
        """Get the optional label of the given source.

        Args:
            source (str): The source to get the label for.

        Returns:
            str: The optional label of the given source.
        """
        source = self._properties.get(source, {})
        if source is None:
            return source
        return source.get(self._LABEL_KEY)

    def __str__(self):
        """Special method to return a properly formatted string representation of the synchronizer."""
        return ("Synchronizer(" +
                "timeout_seconds=" + str(self._timeout_seconds) + ", " +
                "buffer_size=" + str(self._buffer_size) + ", " +
                "allow_zero_events=" + str(self._allow_zero_events) + ", " +
                "actual_properties=" + str(self.actual_properties) + ")")
